#!/usr/bin/perl -w
use strict;
use warnings;
use diagnostics;

use Test::More tests => 78;
use Test::Output;
use Test::Exception;
use Test::File;
use Path::Tiny;

# *** Read before running tests ***
# Warning: While running project "SLUB_SIP_Builder" two directories "export_dir_kitodo" and "tmp/mysip" will be created.
# Start scipt in the project directory to run all tests properly. 

### prepare
BEGIN {
    use Path::Tiny;
    push @INC, Path::Tiny::path(__FILE__)->parent->parent->path("bin")->absolute->stringify;
    require "slubsipbuilder.pl";
    $INC{'SLUB/LZA/SIPBuilder.pm'} = 1; # needed because inlined module
}
my $unpatched_mods=<<'UNPATCHED_MODS';
<?xml version="1.0" encoding="UTF-8"?>
<mods xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns="http://www.loc.gov/mods/v3" version="3.6" xsi:schemaLocation="http://www.loc.gov/mods/v3 http://www.loc.gov/standards/mods/v3/mods-3-6.xsd"><titleInfo><nonSort xml:space="preserve">Der  </nonSort><title>Fichtelberg</title><subTitle>Berg der unbekannten Rekorrde</subTitle></titleInfo><name type="personal"><namePart>Schneider, Dirk</namePart><role><roleTerm type="text">FilmemacherIn</roleTerm></role><role><roleTerm authority="marcrelator" type="code">fmk</roleTerm></role><nameIdentifier>(DE-627)1235502279 (DE-576)165502274</nameIdentifier></name><typeOfResource>moving image</typeOfResource><genre authority="rdacontent">zweidimensionales bewegtes Bild</genre><genre authority="gnd-content">Film</genre><originInfo><place><placeTerm type="code" authority="marccountry">xx</placeTerm></place><dateIssued encoding="marc">2014</dateIssued><issuance>monographic</issuance></originInfo><originInfo eventType="publication"><place><placeTerm type="text">[Leipzig]</placeTerm></place><publisher>top ten tv</publisher><dateIssued>[2014]</dateIssued></originInfo><language><languageTerm authority="iso639-2b" type="code">ger</languageTerm></language><physicalDescription><form authority="marccategory">electronic resource</form><form authority="marcsmd">remote</form><extent>1 Online-Ressource (1 Videodatei, 29:49) farbig</extent><form type="media" authority="rdamedia">Computermedien</form><form type="carrier" authority="rdacarrier">Online-Ressource</form></physicalDescription><targetAudience authority="marctarget">juvenile</targetAudience><note type="statement of responsibility" altRepGroup="00">ein Film von Dirk Schneider</note><note>Dokumentarfilm. Deutschland. 2014</note><relatedItem type="series"><titleInfo><title>MDR</title></titleInfo></relatedItem><relatedItem type="series"><titleInfo><title>Der Osten - entdecke wo du lebst</title></titleInfo></relatedItem><identifier type="oclc">946544758</identifier><recordInfo><descriptionStandard>rda</descriptionStandard><recordContentSource authority="marcorg">DE-576</recordContentSource><recordCreationDate encoding="marc">160304</recordCreationDate><recordChangeDate encoding="iso8601">20160510144338.0</recordChangeDate><recordIdentifier source="DE-576">457035137</recordIdentifier><recordOrigin>Converted from MARCXML to MODS version 3.6 using MARC21slim2MODS3-6.xsl
				(Revision 1.119 2018/06/21)</recordOrigin><languageOfCataloging><languageTerm authority="iso639-2b" type="code">ger</languageTerm></languageOfCataloging></recordInfo></mods>
UNPATCHED_MODS
my $patched_mods=<<'PATCHED_MODS';
<?xml version="1.0" encoding="UTF-8"?>
<mods xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns="http://www.loc.gov/mods/v3" version="3.6" xsi:schemaLocation="http://www.loc.gov/mods/v3 http://www.loc.gov/standards/mods/v3/mods-3-6.xsd"><titleInfo><nonSort xml:space="preserve">Der  </nonSort><title>Fichtelberg</title><subTitle>Berg der unbekannten Rekorrde</subTitle></titleInfo><name type="personal"><namePart>Schneider, Dirk</namePart><role><roleTerm type="text">FilmemacherIn</roleTerm></role><role><roleTerm authority="marcrelator" type="code">fmk</roleTerm></role><nameIdentifier>(DE-627)1235502279 (DE-576)165502274</nameIdentifier></name><typeOfResource>moving image</typeOfResource><genre authority="rdacontent">zweidimensionales bewegtes Bild</genre><genre authority="gnd-content">Film</genre><originInfo><place><placeTerm type="code" authority="marccountry">xx</placeTerm></place><dateIssued encoding="marc">2014</dateIssued><issuance>monographic</issuance></originInfo><originInfo eventType="publication"><place><placeTerm type="text">[Leipzig]</placeTerm></place><publisher>top ten tv</publisher><dateIssued>[2014]</dateIssued></originInfo><language><languageTerm authority="iso639-2b" type="code">ger</languageTerm></language><physicalDescription><form authority="marccategory">electronic resource</form><form authority="marcsmd">remote</form><extent>1 Online-Ressource (1 Videodatei, 29:49) farbig</extent><form type="media" authority="rdamedia">Computermedien</form><form type="carrier" authority="rdacarrier">Online-Ressource</form></physicalDescription><targetAudience authority="marctarget">juvenile</targetAudience><note type="statement of responsibility" altRepGroup="00">ein Film von Dirk Schneider</note><note>Dokumentarfilm. Deutschland. 2014</note><relatedItem type="series"><titleInfo><title>MDR</title></titleInfo></relatedItem><relatedItem type="series"><titleInfo><title>Der Osten - entdecke wo du lebst</title></titleInfo></relatedItem><identifier type="oclc">946544758</identifier><recordInfo><descriptionStandard>rda</descriptionStandard><recordContentSource authority="marcorg">DE-576</recordContentSource><recordCreationDate encoding="marc">160304</recordCreationDate><recordChangeDate encoding="iso8601">20160510144338.0</recordChangeDate><recordIdentifier source="DE-576">457035137</recordIdentifier><recordOrigin>Converted from MARCXML to MODS version 3.6 using MARC21slim2MODS3-6.xsl
				(Revision 1.119 2018/06/21)</recordOrigin><languageOfCataloging><languageTerm authority="iso639-2b" type="code">ger</languageTerm></languageOfCataloging></recordInfo></mods>
PATCHED_MODS
my $unpatched_mods_obj = XML::LibXML->load_xml(string => $unpatched_mods);

my $unpatched_marcblob=<<'UNPATCHED_MARCBLOB';
<?xml version="1.0"?>
<record xmlns="http://www.loc.gov/MARC21/slim">
  <leader>     cgm a22        4500</leader>
  <controlfield tag="001">457035137</controlfield>
  <controlfield tag="003">DE-576</controlfield>
  <controlfield tag="005">20160510144338.0</controlfield>
  <controlfield tag="006">m     o  | |      </controlfield>
  <controlfield tag="007">cr uuu---uuuuu</controlfield>
  <controlfield tag="007">vu uuuuuu</controlfield>
  <controlfield tag="008">160304s2014    xx ger c</controlfield>
  <datafield tag="035" ind1=" " ind2=" ">
    <subfield code="a">(DE-627)1655506501</subfield>
  </datafield>
  <datafield tag="035" ind1=" " ind2=" ">
    <subfield code="a">(DE-576)457035137</subfield>
  </datafield>
  <datafield tag="035" ind1=" " ind2=" ">
    <subfield code="a">(DE-599)BSZ457035137</subfield>
  </datafield>
  <datafield tag="035" ind1=" " ind2=" ">
    <subfield code="a">(OCoLC)946544758</subfield>
  </datafield>
  <datafield tag="040" ind1=" " ind2=" ">
    <subfield code="a">DE-576</subfield>
    <subfield code="b">ger</subfield>
    <subfield code="c">DE-576</subfield>
    <subfield code="e">rda</subfield>
  </datafield>
  <datafield tag="041" ind1=" " ind2=" ">
    <subfield code="a">ger</subfield>
  </datafield>
  <datafield tag="245" ind1="1" ind2="4">
    <subfield code="a">Der Fichtelberg</subfield>
    <subfield code="b">Berg der unbekannten Rekorrde</subfield>
    <subfield code="c">ein Film von Dirk Schneider</subfield>
  </datafield>
  <datafield tag="264" ind1=" " ind2="1">
    <subfield code="a">[Leipzig]</subfield>
    <subfield code="b">top ten tv</subfield>
    <subfield code="c">[2014]</subfield>
  </datafield>
  <datafield tag="264" ind1=" " ind2="4">
    <subfield code="c">&#xA9; 2014</subfield>
  </datafield>
  <datafield tag="300" ind1=" " ind2=" ">
    <subfield code="a">1 Online-Ressource (1 Videodatei, 29:49)</subfield>
    <subfield code="b">farbig</subfield>
  </datafield>
  <datafield tag="336" ind1=" " ind2=" ">
    <subfield code="a">zweidimensionales bewegtes Bild</subfield>
    <subfield code="b">tdi</subfield>
    <subfield code="2">rdacontent</subfield>
  </datafield>
  <datafield tag="337" ind1=" " ind2=" ">
    <subfield code="a">Computermedien</subfield>
    <subfield code="b">c</subfield>
    <subfield code="2">rdamedia</subfield>
  </datafield>
  <datafield tag="338" ind1=" " ind2=" ">
    <subfield code="a">Online-Ressource</subfield>
    <subfield code="b">cr</subfield>
    <subfield code="2">rdacarrier</subfield>
  </datafield>
  <datafield tag="490" ind1="0" ind2=" ">
    <subfield code="a">MDR</subfield>
  </datafield>
  <datafield tag="490" ind1="0" ind2=" ">
    <subfield code="a">Der Osten - entdecke wo du lebst</subfield>
  </datafield>
  <datafield tag="500" ind1=" " ind2=" ">
    <subfield code="a">Dokumentarfilm. Deutschland. 2014</subfield>
  </datafield>
  <datafield tag="591" ind1=" " ind2=" ">
    <subfield code="a">Fernsehmitschnitt (SWB)</subfield>
  </datafield>
  <datafield tag="655" ind1=" " ind2="7">
    <subfield code="a">Film</subfield>
    <subfield code="0">(DE-588)4017102-4</subfield>
    <subfield code="0">(DE-627)104559683</subfield>
    <subfield code="0">(DE-576)208918531</subfield>
    <subfield code="2">gnd-content</subfield>
  </datafield>
  <datafield tag="700" ind1="1" ind2=" ">
    <subfield code="a">Schneider, Dirk</subfield>
    <subfield code="e">FilmemacherIn</subfield>
    <subfield code="0">(DE-627)1235502279</subfield>
    <subfield code="0">(DE-576)165502274</subfield>
    <subfield code="4">fmk</subfield>
  </datafield>
  <datafield tag="935" ind1=" " ind2=" ">
    <subfield code="c">vide</subfield>
  </datafield>
  <datafield tag="937" ind1=" " ind2=" ">
    <subfield code="a">Dokumentarfilm</subfield>
    <subfield code="b">Deutschland</subfield>
    <subfield code="c">2014</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">000  xxxxxcx  a22     zn 4500</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">001 901795887</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">003  DE-576</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">004  457035137</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">005  20160510125331</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">008  160304||||||||||||||||ger|||||||</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">040  </subfield>
    <subfield code="a">DE-14</subfield>
    <subfield code="c">DE-576</subfield>
    <subfield code="d">DE-14</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">852  </subfield>
    <subfield code="z">Fernsehmitschnitt: MDR, 04.02.2014. - Beilage</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">852  </subfield>
    <subfield code="a">DE-14</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">852 1</subfield>
    <subfield code="9">00</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">866  </subfield>
    <subfield code="x">ddsu,pn</subfield>
  </datafield>
</record>
UNPATCHED_MARCBLOB

my $patched_marcblob =<<'PATCHED_MARCBLOB';
<?xml version="1.0"?>
<record xmlns="http://www.loc.gov/MARC21/slim">
  <leader>     cgm a22        4500</leader>
  <controlfield tag="001">457035137</controlfield>
  <controlfield tag="003">DE-576</controlfield>
  <controlfield tag="005">20160510144338.0</controlfield>
  <controlfield tag="006">m     o  | |      </controlfield>
  <controlfield tag="007">cr uuu---uuuuu</controlfield>
  <controlfield tag="007">vu uuuuuu</controlfield>
  <controlfield tag="008">160304s2014    xx ger c</controlfield>
  <datafield tag="035" ind1=" " ind2=" ">
    <subfield code="a">(DE-627)1655506501</subfield>
  </datafield>
  <datafield tag="035" ind1=" " ind2=" ">
    <subfield code="a">(DE-576)457035137</subfield>
  </datafield>
  <datafield tag="035" ind1=" " ind2=" ">
    <subfield code="a">(DE-599)BSZ457035137</subfield>
  </datafield>
  <datafield tag="035" ind1=" " ind2=" ">
    <subfield code="a">(OCoLC)946544758</subfield>
  </datafield>
  <datafield tag="040" ind1=" " ind2=" ">
    <subfield code="a">DE-576</subfield>
    <subfield code="b">ger</subfield>
    <subfield code="c">DE-576</subfield>
    <subfield code="e">rda</subfield>
  </datafield>
  <datafield tag="041" ind1=" " ind2=" ">
    <subfield code="a">ger</subfield>
  </datafield>
  <datafield tag="245" ind1="1" ind2="4">
    <subfield code="a">Der Fichtelberg</subfield>
    <subfield code="b">Berg der unbekannten Rekorrde</subfield>
    <subfield code="c">ein Film von Dirk Schneider</subfield>
  </datafield>
  <datafield tag="264" ind1=" " ind2="1">
    <subfield code="a">[Leipzig]</subfield>
    <subfield code="b">top ten tv</subfield>
    <subfield code="c">[2014]</subfield>
  </datafield>
  <datafield tag="264" ind1=" " ind2="4">
    <subfield code="c">&#xA9; 2014</subfield>
  </datafield>
  <datafield tag="300" ind1=" " ind2=" ">
    <subfield code="a">1 Online-Ressource (1 Videodatei, 29:49)</subfield>
    <subfield code="b">farbig</subfield>
  </datafield>
  <datafield tag="336" ind1=" " ind2=" ">
    <subfield code="a">zweidimensionales bewegtes Bild</subfield>
    <subfield code="b">tdi</subfield>
    <subfield code="2">rdacontent</subfield>
  </datafield>
  <datafield tag="337" ind1=" " ind2=" ">
    <subfield code="a">Computermedien</subfield>
    <subfield code="b">c</subfield>
    <subfield code="2">rdamedia</subfield>
  </datafield>
  <datafield tag="338" ind1=" " ind2=" ">
    <subfield code="a">Online-Ressource</subfield>
    <subfield code="b">cr</subfield>
    <subfield code="2">rdacarrier</subfield>
  </datafield>
  <datafield tag="490" ind1="0" ind2=" ">
    <subfield code="a">MDR</subfield>
  </datafield>
  <datafield tag="490" ind1="0" ind2=" ">
    <subfield code="a">Der Osten - entdecke wo du lebst</subfield>
  </datafield>
  <datafield tag="500" ind1=" " ind2=" ">
    <subfield code="a">Dokumentarfilm. Deutschland. 2014</subfield>
  </datafield>
  <datafield tag="591" ind1=" " ind2=" ">
    <subfield code="a">Fernsehmitschnitt (SWB)</subfield>
  </datafield>
  <datafield tag="655" ind1=" " ind2="7">
    <subfield code="a">Film</subfield>
    <subfield code="0">(DE-588)4017102-4</subfield>
    <subfield code="0">(DE-627)104559683</subfield>
    <subfield code="0">(DE-576)208918531</subfield>
    <subfield code="2">gnd-content</subfield>
  </datafield>
  <datafield tag="700" ind1="1" ind2=" ">
    <subfield code="a">Schneider, Dirk</subfield>
    <subfield code="e">FilmemacherIn</subfield>
    <subfield code="0">(DE-627)1235502279</subfield>
    <subfield code="0">(DE-576)165502274</subfield>
    <subfield code="4">fmk</subfield>
  </datafield>
  <datafield tag="935" ind1=" " ind2=" ">
    <subfield code="c">vide</subfield>
  </datafield>
  <datafield tag="937" ind1=" " ind2=" ">
    <subfield code="a">Dokumentarfilm</subfield>
    <subfield code="b">Deutschland</subfield>
    <subfield code="c">2014</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">000  xxxxxcx  a22     zn 4500</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">001 901795887</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">003  DE-576</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">004  457035137</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">005  20160510125331</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">008  160304||||||||||||||||ger|||||||</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">040  </subfield>
    <subfield code="a">DE-14</subfield>
    <subfield code="c">DE-576</subfield>
    <subfield code="d">DE-14</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">852  </subfield>
    <subfield code="z">Fernsehmitschnitt: MDR, 04.02.2014. - Beilage</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">852  </subfield>
    <subfield code="a">DE-14</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">852 1</subfield>
    <subfield code="9">00</subfield>
  </datafield>
  <datafield tag="LOK" ind1=" " ind2=" ">
    <subfield code="0">866  </subfield>
    <subfield code="x">ddsu,pn</subfield>
  </datafield>
</record>
PATCHED_MARCBLOB
my $unpatched_marcblob_obj = XML::LibXML->load_xml(string => $unpatched_marcblob);

my $unpatched_marc21slimutils = <<'UNPATCHED_MARC21SLIMUTIL';
<?xml version='1.0'?>
<xsl:stylesheet version="1.0" xmlns:marc="http://www.loc.gov/MARC21/slim"
	xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

	<!-- 08/08/08: tmee added corrected chopPunctuation templates for 260c -->
	<!-- 08/19/04: ntra added "marc:" prefix to datafield element -->
	<!-- 12/14/07: ntra added url encoding template -->
	<!-- url encoding -->

	<xsl:variable name="ascii">
		<xsl:text> !"#$%&amp;'()*+,-./0123456789:;&lt;=&gt;?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\]^_`abcdefghijklmnopqrstuvwxyz{|}~</xsl:text>
	</xsl:variable>

	<xsl:variable name="latin1">
		<xsl:text> ¡¢£¤¥¦§¨©ª«¬­®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖ×ØÙÚÛÜÝÞßàáâãäåæçèéêëìíîïðñòóôõö÷øùúûüýþÿ</xsl:text>
	</xsl:variable>
	<!-- Characters that usually don't need to be escaped -->
	<xsl:variable name="safe">
		<xsl:text>!'()*-.0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz~</xsl:text>
	</xsl:variable>

	<xsl:variable name="hex">0123456789ABCDEF</xsl:variable>


	<xsl:template name="datafield">
		<xsl:param name="tag"/>
		<xsl:param name="ind1">
			<xsl:text> </xsl:text>
		</xsl:param>
		<xsl:param name="ind2">
			<xsl:text> </xsl:text>
		</xsl:param>
		<xsl:param name="subfields"/>
		<xsl:element name="marc:datafield">
			<xsl:attribute name="tag">
				<xsl:value-of select="$tag"/>
			</xsl:attribute>
			<xsl:attribute name="ind1">
				<xsl:value-of select="$ind1"/>
			</xsl:attribute>
			<xsl:attribute name="ind2">
				<xsl:value-of select="$ind2"/>
			</xsl:attribute>
			<xsl:copy-of select="$subfields"/>
		</xsl:element>
	</xsl:template>

	<xsl:template name="subfieldSelect">
		<xsl:param name="codes">abcdefghijklmnopqrstuvwxyz</xsl:param>
		<xsl:param name="delimeter">
			<xsl:text> </xsl:text>
		</xsl:param>
		<xsl:variable name="str">
			<xsl:for-each select="marc:subfield">
				<xsl:if test="contains($codes, @code)">
					<xsl:value-of select="text()"/>
					<xsl:value-of select="$delimeter"/>
				</xsl:if>
			</xsl:for-each>
		</xsl:variable>
		<xsl:value-of select="substring($str,1,string-length($str)-string-length($delimeter))"/>
	</xsl:template>

	<xsl:template name="buildSpaces">
		<xsl:param name="spaces"/>
		<xsl:param name="char">
			<xsl:text> </xsl:text>
		</xsl:param>
		<xsl:if test="$spaces>0">
			<xsl:value-of select="$char"/>
			<xsl:call-template name="buildSpaces">
				<xsl:with-param name="spaces" select="$spaces - 1"/>
				<xsl:with-param name="char" select="$char"/>
			</xsl:call-template>
		</xsl:if>
	</xsl:template>

	<xsl:template name="chopPunctuation">
		<xsl:param name="chopString"/>
		<xsl:param name="punctuation">
			<xsl:text>.:,;/ </xsl:text>
		</xsl:param>
		<xsl:variable name="length" select="string-length($chopString)"/>
		<xsl:choose>
			<xsl:when test="$length=0"/>
			<xsl:when test="contains($punctuation, substring($chopString,$length,1))">
				<xsl:call-template name="chopPunctuation">
					<xsl:with-param name="chopString" select="substring($chopString,1,$length - 1)"/>
					<xsl:with-param name="punctuation" select="$punctuation"/>
				</xsl:call-template>
			</xsl:when>
			<xsl:when test="not($chopString)"/>
			<xsl:otherwise>
				<xsl:value-of select="$chopString"/>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>

	<xsl:template name="chopPunctuationFront">
		<xsl:param name="chopString"/>
		<xsl:variable name="length" select="string-length($chopString)"/>
		<xsl:choose>
			<xsl:when test="$length=0"/>
			<xsl:when test="contains('.:,;/[ ', substring($chopString,1,1))">
				<xsl:call-template name="chopPunctuationFront">
					<xsl:with-param name="chopString" select="substring($chopString,2,$length - 1)"
					/>
				</xsl:call-template>
			</xsl:when>
			<xsl:when test="not($chopString)"/>
			<xsl:otherwise>
				<xsl:value-of select="$chopString"/>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>

	<xsl:template name="chopPunctuationBack">
		<xsl:param name="chopString"/>
		<xsl:param name="punctuation">
			<xsl:text>.:,;/] </xsl:text>
		</xsl:param>
		<xsl:variable name="length" select="string-length($chopString)"/>
		<xsl:choose>
			<xsl:when test="$length=0"/>
			<xsl:when test="contains($punctuation, substring($chopString,$length,1))">
				<xsl:call-template name="chopPunctuation">
					<xsl:with-param name="chopString" select="substring($chopString,1,$length - 1)"/>
					<xsl:with-param name="punctuation" select="$punctuation"/>
				</xsl:call-template>
			</xsl:when>
			<xsl:when test="not($chopString)"/>
			<xsl:otherwise>
				<xsl:value-of select="$chopString"/>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>

	<!-- nate added 12/14/2007 for lccn.loc.gov: url encode ampersand, etc. -->
	<xsl:template name="url-encode">

		<xsl:param name="str"/>

		<xsl:if test="$str">
			<xsl:variable name="first-char" select="substring($str,1,1)"/>
			<xsl:choose>
				<xsl:when test="contains($safe,$first-char)">
					<xsl:value-of select="$first-char"/>
				</xsl:when>
				<xsl:otherwise>
					<xsl:variable name="codepoint">
						<xsl:choose>
							<xsl:when test="contains($ascii,$first-char)">
								<xsl:value-of
									select="string-length(substring-before($ascii,$first-char)) + 32"
								/>
							</xsl:when>
							<xsl:when test="contains($latin1,$first-char)">
								<xsl:value-of
									select="string-length(substring-before($latin1,$first-char)) + 160"/>
								<!-- was 160 -->
							</xsl:when>
							<xsl:otherwise>
								<xsl:message terminate="no">Warning: string contains a character
									that is out of range! Substituting "?".</xsl:message>
								<xsl:text>63</xsl:text>
							</xsl:otherwise>
						</xsl:choose>
					</xsl:variable>
					<xsl:variable name="hex-digit1"
						select="substring($hex,floor($codepoint div 16) + 1,1)"/>
					<xsl:variable name="hex-digit2" select="substring($hex,$codepoint mod 16 + 1,1)"/>
					<!-- <xsl:value-of select="concat('%',$hex-digit2)"/> -->
					<xsl:value-of select="concat('%',$hex-digit1,$hex-digit2)"/>
				</xsl:otherwise>
			</xsl:choose>
			<xsl:if test="string-length($str) &gt; 1">
				<xsl:call-template name="url-encode">
					<xsl:with-param name="str" select="substring($str,2)"/>
				</xsl:call-template>
			</xsl:if>
		</xsl:if>
	</xsl:template>
</xsl:stylesheet>
<!-- Stylus Studio meta-information - (c)1998-2002 eXcelon Corp.
<metaInformation>
<scenarios/><MapperInfo srcSchemaPath="" srcSchemaRoot="" srcSchemaPathIsRelative="yes" srcSchemaInterpretAsXML="no" destSchemaPath="" destSchemaRoot="" destSchemaPathIsRelative="yes" destSchemaInterpretAsXML="no"/>
</metaInformation>
-->
UNPATCHED_MARC21SLIMUTIL

my $structmap_res = "<mets:structMap TYPE=\"PHYSICAL\">
  <mets:div ID=\"PHYS_0000\" TYPE=\"ieDir\">
  <mets:div ID=\"PHYS_000000000000000_LZA\" TYPE=\"fileorderSequence\">
<mets:fptr FILEID=\"FILE_000000000000000_LZA\" />
</mets:div>
  </mets:div>
</mets:structMap>
";

my $filesec_res = "<mets:fileSec>
  <mets:fileGrp USE=\"LZA\">
  <mets:file ID=\"FILE_000000000000000_LZA\" CHECKSUMTYPE=\"MD5\" CHECKSUM=\"d41d8cd98f00b204e9800998ecf8427e\">
<mets:FLocat xmlns:xlink=\"http://www.w3.org/1999/xlink\" LOCTYPE=\"URL\" xlink:href=\"file://data/test.txt\"/>
</mets:file>
  </mets:fileGrp>
</mets:fileSec>
";

my $noppn_dmd_res = "<mets:dmdSec ID=\"DMDLOG_0000\">
  <!-- bibliographic metadata -->
  <mets:mdWrap MDTYPE=\"MODS\">
    <mets:xmlData>
      <mods version=\"3.6\"
    xmlns=\"http://www.loc.gov/mods/v3\"
    xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"
    xsi:schemaLocation=\"http://www.loc.gov/mods/v3 http://www.loc.gov/standards/mods/v3/mods-3-6.xsd\">
    <identifier>string</identifier>
</mods>

    </mets:xmlData>
  </mets:mdWrap>
</mets:dmdSec>
";

# preparation
my $xsl_path = path(__FILE__)->parent->parent->child('xsl');
my $export_dir_kitodo_path = path(__FILE__)->parent->parent->child('export_dir_kitodo');
my $valid_ie_dir = "10_00-8";
my $not_valid_ie_dir = "10_00-8+";
my $valid_ie_dir_path = $export_dir_kitodo_path->absolute . "/" . $valid_ie_dir;
my $not_valid_ie_dir_path = $export_dir_kitodo_path->absolute . "/" . $not_valid_ie_dir;
my $file = "test.txt";
my $file2 = "test+.txt";
if(! -d $export_dir_kitodo_path) { mkdir $export_dir_kitodo_path or die "Error creating directory: $export_dir_kitodo_path"; }
if(! -d $export_dir_kitodo_path->child($valid_ie_dir)) { mkdir $export_dir_kitodo_path->child($valid_ie_dir) or die "Error creating directory: $export_dir_kitodo_path->child($valid_ie_dir)"; }
if(! -f $export_dir_kitodo_path->child($valid_ie_dir)->child($file)) { 
	open FILE, '>'.$export_dir_kitodo_path->child($valid_ie_dir)->child($file) or die "Unable to create $file"; 
	close FILE; 
}	
if(! -d $export_dir_kitodo_path->child($not_valid_ie_dir)) { mkdir $export_dir_kitodo_path->child($not_valid_ie_dir) or die "Error creating directory: $export_dir_kitodo_path->child($not_valid_ie_dir)"; }
if(! -f $export_dir_kitodo_path->child($not_valid_ie_dir)->child($file2)) { 
	open FILE, '>'.$export_dir_kitodo_path->child($not_valid_ie_dir)->child($file2) or die "Unable to create $file"; 
	close FILE; 
}

my $mysip_path = path(__FILE__)->parent->parent->child('tmp')->child('mysip');
if(! -d $mysip_path->parent) { mkdir $mysip_path->parent or die "Error creating directory: tmp";}
if(! -d $mysip_path) { mkdir $mysip_path or die "Error creating directory: $mysip_path";}

#~ structure of input output folders
#~ .
#~ ├── export_dir_kitodo
#~ │   ├── 10_00-8
#~ │   │   └── test.txt
#~ │   │   
#~ │   └── 10_00-8+
#~ │	   └── test+.txt
#~ └── tmp
#~	   └── mysip

my $ua = LWP::UserAgent->new;
my $request = HTTP::Request->new('GET' => "https://sru.bsz-bw.de/swbf");
my $response = $ua->request($request);
my $marc21slimutils_file = $xsl_path->child('MARC21slimUtils.xsl');
my $useragent_obj = LWP::UserAgent->new;
$useragent_obj->agent("MyApp/0.1 ");
$useragent_obj->timeout(3600); #1h
my $filecopyhash = SLUB::LZA::SIPBuilder::create_filecopyhash($valid_ie_dir_path, $mysip_path->absolute . "/PPN-457035137_2019-10-14_08-46-50/data");
# ensure no dir exists, then run test
if ($xsl_path->is_dir) { $xsl_path->remove_tree; }

### tests
BEGIN { use_ok("SLUB::LZA::SIPBuilder"); }
#
is(SLUB::LZA::SIPBuilder::check_xsl_directory(), $xsl_path->absolute, "check_xsl_directory(), return value if not exist");
ok($xsl_path->is_dir, "check_xsl_directory(), created if not exist");
is(SLUB::LZA::SIPBuilder::check_xsl_directory(), $xsl_path->absolute, "check_xsl_directory(), return value if exist");
ok($xsl_path->is_dir, "check_xsl_directory(), untouched if exist");
#
SKIP: {
	skip "No response from server https://sru.bsz-bw.de/swb", 1 unless ! $response->is_error;
	like(SLUB::LZA::SIPBuilder::get_mods_from("https://sru.bsz-bw.de/swb", "457035137", "pica.swn", "marcxmlvbos"), qr//, "get_mods_from()");
};
throws_ok{ SLUB::LZA::SIPBuilder::get_mods_from("", "457035137", "pica.swn", "marcxmlvbos"); } qr/invalid parameters/, "get_mods_from(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::get_mods_from("https://sru.bsz-bw.de/swb", "", "pica.swn", "marcxmlvbos"); } qr/invalid parameters/, "get_mods_from(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::get_mods_from("https://sru.bsz-bw.de/swb", "457035137", "", "marcxmlvbos"); } qr/invalid parameters/, "get_mods_from(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::get_mods_from("https://sru.bsz-bw.de/swb", "457035137", "pica.swn", ""); } qr/invalid parameters/, "get_mods_from(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::get_mods_from(undef, "457035137", "pica.swn", "marcxmlvbos"); } qr/url not defined/, "get_mods_from(\$url, \$ppn, \$key, \$schema), url undef";
throws_ok{ SLUB::LZA::SIPBuilder::get_mods_from("https://sru.bsz-bw.de/swb", undef, "pica.swn", "marcxmlvbos"); } qr/ppn not defined/, "get_mods_from(\$url, \$ppn, \$key, \$schema), ppn undef";
throws_ok{ SLUB::LZA::SIPBuilder::get_mods_from("https://sru.bsz-bw.de/swb", "457035137", undef, "marcxmlvbos"); } qr/key not defined/, "get_mods_from(\$url, \$ppn, \$key, \$schema), key undef";
throws_ok{ SLUB::LZA::SIPBuilder::get_mods_from("https://sru.bsz-bw.de/swb", "457035137", "pica.swn", undef); } qr/schema not defined/, "get_mods_from(\$url, \$ppn, \$key, \$schema), schema undef";
#
ok(SLUB::LZA::SIPBuilder::write_file($marc21slimutils_file, $unpatched_marc21slimutils), "write_file()");
throws_ok{ SLUB::LZA::SIPBuilder::write_file("", $unpatched_marc21slimutils); } qr/invalid parameters/, "write_file(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::write_file($marc21slimutils_file, ""); } qr/invalid parameters/, "write_file(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::write_file(undef, $unpatched_marc21slimutils); } qr/filename not defined/, "write_file(\$filename, \$value), filename undef";
throws_ok{ SLUB::LZA::SIPBuilder::write_file($marc21slimutils_file, undef); } qr/value not defined/, "write_file(\$filename, \$value), value undef";
#
is(SLUB::LZA::SIPBuilder::patch_mods($unpatched_mods_obj), $patched_mods, "patch_mods()");
throws_ok{ SLUB::LZA::SIPBuilder::patch_mods(""); } qr/invalid parameters/, "patch_mods(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::patch_mods(undef); } qr/modsobject not defined/, "patch_mods(\$modsobj), modsobj undef";
#
is(SLUB::LZA::SIPBuilder::patch_marc_response($unpatched_marcblob_obj), $patched_marcblob, "patch_marc_response()");
throws_ok{ SLUB::LZA::SIPBuilder::patch_marc_response(""); } qr/invalid parameters/, "patch_marc_response(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::patch_marc_response(undef); } qr/marcobject not defined/, "patch_marc_response(\$marcobj), marcobj undef";
#
is(SLUB::LZA::SIPBuilder::check_marc21_utility($xsl_path, $useragent_obj), $marc21slimutils_file , "check_marc21_utility()");
throws_ok{ SLUB::LZA::SIPBuilder::check_marc21_utility("", $useragent_obj); } qr/invalid parameters/, "check_marc21_utility(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::check_marc21_utility($xsl_path, ""); } qr/invalid parameters/, "check_marc21_utility(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::check_marc21_utility(undef, $useragent_obj); } qr/xsl directory not defined/, "check_marc21_utility(\$xsl_dir, \$useragent), xsl_dir undef";
throws_ok{ SLUB::LZA::SIPBuilder::check_marc21_utility($xsl_path, undef); } qr/user agent not defined/, "check_marc21_utility(\$xsl_dir, \$useragent), useragent undef";
#
is(SLUB::LZA::SIPBuilder::check_marc21_mods_xsl($xsl_path, $useragent_obj), 'xsl/MARC21slim2MODS3-6.patched.xsl' , "check_marc21_mods_xsl()");
throws_ok{ SLUB::LZA::SIPBuilder::check_marc21_mods_xsl("", $useragent_obj); } qr/invalid parameters/, "check_marc21_mods_xsl(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::check_marc21_mods_xsl($xsl_path, ""); } qr/invalid parameters/, "check_marc21_mods_xsl(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::check_marc21_mods_xsl(undef, $useragent_obj); } qr/xsl directory not defined/, "check_marc21_mods_xsl(\$xsl_dir, \$useragent), xsl_dir undef";
throws_ok{ SLUB::LZA::SIPBuilder::check_marc21_mods_xsl($xsl_path, undef); } qr/user agent not defined/, "check_marc21_mods_xsl(\$xsl_dir, \$useragent), useragent undef";
#
like(SLUB::LZA::SIPBuilder::create_filecopyhash($valid_ie_dir_path, $mysip_path->absolute . "/PPN-457035137_2019-10-14_08-46-50/data"), qr/HASH/, "create_filecopyhash()");
throws_ok{ SLUB::LZA::SIPBuilder::create_filecopyhash("", $mysip_path->absolute . "/PPN-457035137_2019-10-14_08-46-50/data"); } qr/invalid parameters/, "create_filecopyhash(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::create_filecopyhash($valid_ie_dir_path, ""); } qr/invalid parameters/, "create_filecopyhash(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::create_filecopyhash($valid_ie_dir_path, undef); } qr/content not defined/, "create_filecopyhash(\$directory, \$content), content undef";
throws_ok{ SLUB::LZA::SIPBuilder::create_filecopyhash(undef, $mysip_path->absolute . "/PPN-number_date_time/data"); } qr/directory not defined/, "create_filecopyhash(\$directory, \$content), directory undef";
#
SKIP: {
	skip "No response from server https://sru.bsz-bw.de/swb", 1 unless ! $response->is_error;
	like(SLUB::LZA::SIPBuilder::prepare_dmd_section_with_ppn("457035137"), qr//, "prepare_dmd_section_with_ppn()");
};
throws_ok{ SLUB::LZA::SIPBuilder::prepare_dmd_section_with_ppn(""); } qr/invalid parameters/, "prepare_dmd_section_with_ppn(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_dmd_section_with_ppn(undef); } qr/ppn not defined/, "prepare_dmd_section_with_ppn(\$ppn), ppn undef";
#
ok(SLUB::LZA::SIPBuilder::prepare_dmd_section_with_noppn("string") eq $noppn_dmd_res, "prepare_dmd_section_with_noppn()");
throws_ok{ SLUB::LZA::SIPBuilder::prepare_files_sections(""); } qr/Can't use string/, "prepare_dmd_section_with_noppn(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_dmd_section_with_noppn(undef); } qr/noppn not defined/, "prepare_dmd_section_with_noppn(\$noppn), noppn undef";
#
like(SLUB::LZA::SIPBuilder::prepare_amd_section("2019-10-14T13:33:56", "kitodo", "10008", "false", "DE-14", "Gesetzlicher Auftrag"), qr//, "prepare_amd_section()");
like(SLUB::LZA::SIPBuilder::prepare_amd_section("2019-10-14T13:33:56", "kitodo", "10008", "false", "", "Gesetzlicher Auftrag"), qr//, "prepare_amd_section() with no externel_isil");
throws_ok{ SLUB::LZA::SIPBuilder::prepare_amd_section("", "kitodo", "10008", "false", "DE-14", "Gesetzlicher Auftrag"); } qr/invalid parameters/, "prepare_amd_section(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_amd_section("2019-10-14T13:33:56", "", "10008", "false", "DE-14", "Gesetzlicher Auftrag"); } qr/invalid parameters/, "prepare_amd_section(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_amd_section("2019-10-14T13:33:56", "kitodo", "", "false", "DE-14", "Gesetzlicher Auftrag"); } qr/invalid parameters/, "prepare_amd_section(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_amd_section("2019-10-14T13:33:56", "kitodo", "10008", "", "DE-14", "Gesetzlicher Auftrag"); } qr/invalid parameters/, "prepare_amd_section(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_amd_section("2019-10-14T13:33:56", "kitodo", "10008", "false", "DE-14", ""); } qr/invalid parameters/, "prepare_amd_section(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_amd_section(undef, "kitodo", "10008", "false", "DE-14", "Gesetzlicher Auftrag"); } qr/export to archive date not defined/, "prepare_amd_section(\$export_to_archive_date, \$external_workflow, \$external_id, \$external_conservation_flag, \$external_isil, \$external_value_descr), export_to_archive_date undef";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_amd_section("2019-10-14T13:33:56", undef, "10008", "false", "DE-14", "Gesetzlicher Auftrag"); } qr/external workflow not defined/, "prepare_amd_section(\$export_to_archive_date, \$external_workflow, \$external_id, \$external_conservation_flag, \$external_isil, \$external_value_descr), external_workflow undef";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_amd_section("2019-10-14T13:33:56", "kitodo", undef, "false", "DE-14", "Gesetzlicher Auftrag"); } qr/external id not defined/, "prepare_amd_section(\$export_to_archive_date, \$external_workflow, \$external_id, \$external_conservation_flag, \$external_isil, \$external_value_descr), external_id undef";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_amd_section("2019-10-14T13:33:56", "kitodo", "10008", undef, "DE-14", "Gesetzlicher Auftrag"); } qr/external conservation flag not defined/, "prepare_amd_section(\$export_to_archive_date, \$external_workflow, \$external_id, \$external_conservation_flag, \$external_isil, \$external_value_descr), external_conservation_flag undef";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_amd_section("2019-10-14T13:33:56", "kitodo", "10008", "false", undef, "Gesetzlicher Auftrag"); } qr/external isil not defined/, "prepare_amd_section(\$export_to_archive_date, \$external_workflow, \$external_id, \$external_conservation_flag, \$external_isil, \$external_value_descr), external_isil undef";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_amd_section("2019-10-14T13:33:56", "kitodo", "10008", "false", "DE-14", undef); } qr/external value description not defined/, "prepare_amd_section(\$export_to_archive_date, \$external_workflow, \$external_id, \$external_conservation_flag, \$external_isil, \$external_value_descr), external_value_descr undef";
#
ok(SLUB::LZA::SIPBuilder::prepare_files_sections($filecopyhash) eq $filesec_res, "prepare_files_sections()");
throws_ok{ SLUB::LZA::SIPBuilder::prepare_files_sections(""); } qr/Can't use string/, "prepare_files_sections(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_files_sections(\$filecopyhash); } qr/Not a HASH reference/, "prepare_files_sections(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_files_sections(undef); } qr/filecopyhash not defined/, "prepare_files_sections(\$filecopyhash), filecopyhash undef";
#
ok(SLUB::LZA::SIPBuilder::prepare_struct_map($filecopyhash) eq $structmap_res, "prepare_struct_map()");
throws_ok{ SLUB::LZA::SIPBuilder::prepare_struct_map(""); } qr/Can't use string/, "prepare_struct_map(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_struct_map(\$filecopyhash); } qr/Not a HASH reference/, "prepare_struct_map(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::prepare_struct_map(undef); } qr/filecopyhash not defined/, "prepare_struct_map(\$filecopyhash), filecopyhash undef";
#
is(SLUB::LZA::SIPBuilder::check_directory($valid_ie_dir_path, $valid_ie_dir), "", "check_directory()");
throws_ok{ SLUB::LZA::SIPBuilder::check_directory("", $valid_ie_dir); } qr/invalid parameters/, "check_directory(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::check_directory($valid_ie_dir_path, ""); } qr/invalid parameters/, "check_directory(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::check_directory(undef, $valid_ie_dir); } qr/fullname not defined/, "check_directory(\$valid_ie_dir_path, \$valid_ie_dir), valid_ie_dir_path undef";
throws_ok{ SLUB::LZA::SIPBuilder::check_directory($valid_ie_dir_path, undef); } qr/entry not defined/, "check_directory(\$valid_ie_dir_path, \$valid_ie_dir), valid_ie_dir undef";
like(SLUB::LZA::SIPBuilder::check_directory($export_dir_kitodo_path->absolute . "/LFvNRFgzZQ6nXnK0wvOXqYGA3jBuHrQWWkw2ngWL/5CglQ5VNjq1kCNTVPhcRDYy59PpQg/0g9XWHBkgunA4FXGefoX5RToHrXgE6cY35wLdviFjE2gAf9mpQSTnC/hpH4omOsAl6I9UfU4HYoo4drqLGjn0QP80SPm6TRWThtKmb6dj7iWm1UfFOo9W19vcCQQIcz/1Aa_080B-", "1Aa_080B-"), qr/expected maximum 255 characters/, "validate_directory(\$valid_ie_dir_path, \$valid_ie_dir), expected valid_ie_dir_path max 255 characters");
like(SLUB::LZA::SIPBuilder::check_directory($not_valid_ie_dir_path, $not_valid_ie_dir), qr/you need to specify/, "check_directory(\$valid_ie_dir_path, \$valid_ie_dir), file or directory must be valid to (^[A-Za-z0-9_.-]+\$)");
like(SLUB::LZA::SIPBuilder::check_directory($valid_ie_dir_path . "/../", $valid_ie_dir), qr/relativ path/, "check_directory(\$valid_ie_dir_path, \$valid_ie_dir), relative path in form ../ is not allowed");
#
is(SLUB::LZA::SIPBuilder::validate_directory($valid_ie_dir_path), "", "validate_directory()");
throws_ok{ SLUB::LZA::SIPBuilder::validate_directory(""); } qr/Could not find/, "validate_directory(), invalid parameters";
throws_ok{ SLUB::LZA::SIPBuilder::validate_directory(undef); } qr/directory not defined/, "validate_directory(\$directory), directory undef";

1;
