#!/usr/bin/env bash

# checks the size of the used Oracle DB segments

# copy plugin to the "/usr/lib/check_mk_agent/plugin" directory
# test it via: check_mk_agent | grep -v grep | grep -A 3 '<<<check_oracle_db_segment_size>>>'

# REQUIREMENTS:
# - Bash 4 or higher
# - Oracle 11 DB or higher with sqlplus installed

scriptname=$( basename "${0}" ".sh" )
LOCKFILE="/var/lock/${scriptname}.lock"

status=3
itemname='oracle_segment_size'
perf_values="-"

# check if sqlplus is installed
BINARY="$(su oracle -c 'command -v sqlplus')"
[[ ${BINARY} ]] || exit 1

# IMPORTANT: Set lock using "flock", NOT "touch"!!! It's atomic and doesn't have to be cleared after the script ran.
(
flock -n 9 || exit 1
	ORACLE_OUTPUT=$( /bin/su - oracle -c"${BINARY} -S / as sysdba <<-EOF
		set heading off;
		set feedback off;
		set trimspool on;
		select sum(bytes) from dba_segments;
		exit;
	EOF")

	ORACLE_OUTPUT=$( echo "${ORACLE_OUTPUT}" | tr -d '[:space:]' )

	if [[ ( "${ORACLE_OUTPUT}" =~ [0-9]* ) ]]; then
		# SQLplus gave a valid answer, all OK
		status=0
		perf_values="segment_bytes=${ORACLE_OUTPUT}"
		statustext=$( echo "scale=2; ${ORACLE_OUTPUT}/1024/1024/1024" | bc )" GB of Oracle DB segments are actually in use."
	else
		status=3
		perf_values="-"
		statustext="Couldn' get used Oracle DB segments."
	fi

	echo "${status} ${itemname} ${perf_values} ${statustext}"
) 9>"${LOCKFILE}"

# Code checked by shellcheck (https://github.com/koalaman/shellcheck) on 2020-12-02
