#!/usr/bin/env bash

# checks the integrity of the Oracle DB tables

# copy plugin to the "/usr/lib/check_mk_agent/plugin" directory
# test it via: check_mk_agent | grep -v grep | grep -A 3 '<<<check_oracle_db_table_integrity>>>'

# REQUIREMENTS:
# - Bash 4 or higher
# - Oracle 11 DB or higher with sqlplus installed

scriptname=$( basename "${0}" ".sh" )
LOCKFILE="/var/lock/${scriptname}.lock"

status=3
itemname='oracle_tables_integrity'
perf_values="-"

# check if sqlplus is installed
BINARY="$(su oracle -c 'command -v sqlplus')"
[[ ${BINARY} ]] || exit 1

create_commandlist(){
	/bin/su - oracle -c"${BINARY} -S / as sysdba <<-EOF
		/*
		    shamelessly stolen (and adapted) from:
		    https://oracle-base.com/dba/script?category=miscellaneous&file=analyze_all.sql
		
		    more documentation:
		    https://www.oracle.com/technetwork/database/availability/corruption-bestpractices-12c-2141348.pdf
		*/
		SET PAGESIZE 0
		SET FEEDBACK OFF
		SET VERIFY OFF
		SET TRIMSPOOL ON
		
		/* write temporary file with all the instructions */
		SPOOL /tmp/${scriptname}.tempsql
		SELECT 'ANALYZE TABLE ' || OWNER || '.' || OBJECT_NAME || ' VALIDATE STRUCTURE CASCADE;'
		        FROM   DBA_OBJECTS
		        WHERE  OBJECT_TYPE = 'TABLE';
		SPOOL OFF
		
		/* run the instructions
		   Comment out following line to prevent immediate run */
		--@/tmp/${scriptname}.tempsql
		
		SET PAGESIZE 14
		SET FEEDBACK ON
		SET VERIFY ON
		exit;
	EOF"

	echo "exit;" >> "/tmp/${scriptname}.tempsql"
}

# IMPORTANT: Set lock using "flock", NOT "touch"!!! It's atomic and doesn't have to be cleared after the script ran.
(
flock -n 9 || exit 1
	if [[ ( ! -e "/tmp/${scriptname}.tempsql" ) && \
 	      ( $(date -r "/tmp/${scriptname}.tempsql" +%s) -lt $(date -d 'now - 14 days' +%s) ) ]]; then
		create_commandlist
	fi

	/bin/su - oracle -c"${BINARY} -S / as sysdba @/tmp/${scriptname}.tempsql > /tmp/${scriptname}.log"

	# THE FOLLOWING ERRORS ARE DETECTED:
	# - ORA-01210: data file header is media corrupt
	#   (https://docs.oracle.com/database/121/ERRMG/ORA-00910.htm#ERRMG-GUID-D9EBDFFA-88C6-4185-BD2C-E1B959A97274)
	# - ORA-01578: ORACLE data block corrupted (file # string, block # string)
	#   (https://docs.oracle.com/database/121/ERRMG/ORA-01500.htm#ERRMG-GUID-65B2B9E5-7075-4D53-91B8-FCAECA0AEE0E)
	# FEEL FREE TO ADD MORE ERRORS AS NECESSARY.
	if [[ $( grep -q "ORA-01210" "/tmp/${scriptname}.log" ) || \
              $( grep -q "ORA-01578" "/tmp/${scriptname}.log" ) ]]; then
		ORACLE_OUTPUT=3
	fi

	if [[ ( ${ORACLE_OUTPUT} -eq 0 ) ]]; then
		status=0
		perf_values="-"
		statustext="Database integrity OK."
	else
		status=3
		perf_values="-"
		statustext="Database integrity broken, please go look into that NOW!"
	fi

	echo "${status} ${itemname} ${perf_values} ${statustext}"
) 9>"${LOCKFILE}"

rm -f "/tmp/${scriptname}.tempsql"
rm -f "/tmp/${scriptname}.log"

# Code checked by shellcheck (https://github.com/koalaman/shellcheck) on 2020-12-02
