package SLUB::LZA::TA;
use v5.36;
use Path::Tiny qw( path );
use YAML qw(LoadFile);
use LWP::UserAgent;
use SOAP::Lite;
use Carp qw( croak );
use Regexp::Optimizer;
use IO::Zlib;
use Text::CSV_PP;
use SLUB::LZA::TA::Crypt;
use namespace::clean;
use App::Cmd::Setup -app;

# VERSION

# ABSTRACT: main module for ta-tool

our %config;
our %cache;
our $cache_path;
our $SALT = pack("H16", "There is no security by obscurity!");

sub load_config_if_possible ($homestr) {
    my $home = path($homestr);
    if ($home->is_dir() && !$home->is_rootdir) {
        my $config_path = $home->child('.config')->child('ta-tool.rc');
        our $config_file = $config_path;
        if ($config_path->is_file) {
            my $config_ref = YAML::LoadFile($config_path);
            if (defined $config_ref->{password}) {
                $config_ref->{decrypted_password} = SLUB::LZA::TA::Crypt::decrypt($config_ref->{password});
            }
            %config = %{$config_ref};
            $config{elasticsearch_protocol} = $config{http_only} ? 'http' : 'https';
        }

    }
    return 1;
}

sub load_cache_if_possible ($homestr) {
    my $home = path($homestr);
    if ($home->is_dir() && !$home->is_rootdir) {
        $cache_path = $home->child('.cache')->child('ta-tool.cache');
        if ($cache_path->is_file and -s $cache_path < 8192 * 1024) {
            # if size > 8MB, write new at end, see END{}-block
            my $cache_ref = YAML::LoadFile($cache_path);
            %cache = %{$cache_ref};
        }
    }
    return 1;
}

BEGIN {
    my $rx_psep = qr{[/\\]};
    my $rx_abs  = qr{[A-Z]:};
    my $rx_sub  = qr{([[:print:]]+)};
    if ($ENV{'HOME'} =~ m{^(($rx_abs)?($rx_psep$rx_sub)+)$}m) {
        load_config_if_possible($1);
        load_cache_if_possible($1);
    }
}

sub _override_config ($override, $opt) {
    if (exists $opt->{$override}) {
        if (!exists $config{$override}) {
            $config{$override} = $opt->{$override};
        }
        if (!defined $config{$override}) {
            $config{$override} = ""; # set temporary to empty string
        }
        if ($config{$override} ne $opt->{$override}) {
            say STDERR <<"HINT";
Hint: override local config with:
    $override => $opt->{$override} (config: $config{$override})
HINT
            $config{$override} = $opt->{$override};
        }
    }
    return 1;
}

sub common_global_validate ($self, $opt, $args) {
    foreach my $override (qw(elasticsearch_host elasticsearch_port http_only permanent_volume)) {
        _override_config($override, $opt);
    }
    # option overrides config
    $config{elasticsearch_protocol} = $config{http_only} ? 'http' : 'https';
    foreach my $volume (@{ $SLUB::LZA::TA::config{permanent_volume} }) {
        if (! -d "$volume") {$self->usage_error("given volume '$volume', does not exist or readable, $!");}
    }
    return 1;
}

sub common_global_opt_spec () {
    return (
        [ "help|h" => "this usage screen" ],
        [],
        [ "debug|D" => "enable user agent debug output" ],
        [ "verbose|v" => "enable verbose output" ],
        [ "elasticsearch-host|e=s" => "host adress where Archivematica's ElasticSearch runs, overrides config value (command 'init')" ],
        [ "elasticsearch-port|E=s" => "port of Archivematica's ElasticSearch, overrides config value (command 'init')", {default => 9200} ],
        [ "http-only" => "use HTTP instead HTTPS, overrides config value (command 'init')"],
        [ "permanent-volume|P=s@" => "path[s] to volume where Archivematica stores AIPs permanently, overrides config value (command 'init'), repeatable)"],
        #[ "logdir|l=s" => "logdir where Archivematica stores it server log files"],
    );
}

END {
    if ($cache_path->parent->is_dir && !$cache_path->parent->parent->is_rootdir && $cache_path->touch()) {
        YAML::DumpFile($cache_path, \%cache)
    } else {
        warn "The $cache_path could not be written, because dir ".$cache_path->parent()." is not writeable";
    }
}

1;
