package SLUB::LZA::TA::Command::count;
use SLUB::LZA::TA -command;
use v5.36;
use SLUB::LZA::TA::Archivematica::Elasticsearch;
use SLUB::LZA::TA::Archivematica::Elasticsearch::PrepareQuery;
use Date::Calc qw(Date_to_Time);
use namespace::autoclean -except => qr{SLUB::LZA::TA::.*};

# VERSION

# ABSTRACT: count IEs module for ta-tool

sub abstract {return "count AIPs in Archival Information System (AIS)";}

my $description=<<"DESCR";
Searches an AIS for descriptive or source metadata on behalf of the Technical Analyst
and return count of matches.

Examples:

  * Is this dc identifier in archive?
    '$0 count -d SLUB:LZA:Kitodo:kitodo:422766'
  * How many AIPs were created in 2021-05-31?
    '$0 count -c 2021-05-31'
  * How many AIPs with mkv-files are in archive?
    '$0 count --pronom-id fmt/569'
DESCR

sub description {
    return "$description"
}
sub opt_spec {
    my @global_opts= SLUB::LZA::TA::common_global_opt_spec();
    my @local_opts = (
        #["source|s=s" => "count based on string search in source metadata"],
        ["aip|a=s" => "count AIPs by given AIP id"],
        [ 'datemode' => hidden => {
            one_of => [
                [ 'creationdate|c=s' => 'count based on creation date in format "YYYY-MM-DD"' ],
                [ 'creationdate-from=s' => 'count based on creation date ranges, beginning date in format "YYYY-MM-DD", implies "--creationdate-to"'],
                #[ 'modificationdate|m=s' => 'search based on modificationdate string' ]
            ]
          }
        ],
        [ 'creationdate-to=s' => 'search based on creation date ranges, beginning date in format "YYYY-MM-DD", implies "--creationdate-from"'],
        ['descriptive|d=s' => 'count based on string search in descriptive metadata, using exact match'],
        ['fuzzy|f=s' => 'count based on string search in descriptive metadata, using phrase prefix match' ],
        ['lzaid|l=s' => 'count AIPs by given LZA id'],
        ['format' => hidden => {one_of => [
            ['pronom-id|p=s' => 'with pronom format id'],
            #['without-format=s' => "without pronom format id"],
        ] } ],
        [ 'onlymode' => hidden => {
            one_of => [
                [ 'only-migrated', 'only if migrated from other AIS' ],
                [ 'only-updated', 'only if AIP is an AIP update' ],
                [ 'only-new|only-first-ingest', 'only if AIP is first ingest' ],
        ] } ],
        [ 'workflow|w=s' => 'LZA internal workflow name'],
        [ 'ldpmode' => hidden => {
            one_of => [
                [ 'only-ldp-saxon', 'only if AIP is LDP Saxon funded' ],
                [ 'only-ldp', 'only if AIP is LDP funded' ],
                [ 'only-ldp-without-saxon', 'only if AIP is LDP but not Saxon funded' ],
                [ 'no-ldp', 'only if AIP is not LDP funded' ],
            ] } ],
    );
    return (@global_opts, [], @local_opts);
}
sub validate_args { ## no critic (CognitiveComplexity::ProhibitExcessCognitiveComplexity)
    my ($self, $opt, $args) = @_;
    SLUB::LZA::TA::common_global_validate($self, $opt, $args);
    # no args allowed but options!
    $self->usage_error("No args allowed") if @$args;
    my $from_epoch;
    my $to_epoch;
    if (exists $opt->{creationdate}) {
        if ($opt->{creationdate} =~ m/^(\d{4})-(\d{2})-(\d{2})$/ ) {
            $from_epoch = Date_to_Time($1, $2, $3, 0, 0, 0);
            $to_epoch = Date_to_Time($1, $2, $3, 23, 59, 59);
        } else {
            $self->usage_error('--creationdate expects date in format "YYYY-MM-DD"');
        }
    }
    if (exists $opt->{creationdate_from}) {
        $self->usage_error('--creationdate-from implies --creationdate-to"') unless exists $opt->{creationdate_to};
        if ($opt->{creationdate_from} =~ m/^(\d{4})-(\d{2})-(\d{2})$/ ) {
            $from_epoch = Date_to_Time($1, $2, $3, 0, 0, 0);
        } else {

            $self->usage_error('--creationdate-from expects date in format "YYYY-MM-DD", got "'.$opt->{creationdate_from}.'"');
        }
    }

    if (exists $opt->{creationdate_to}) {
        $self->usage_error('--creationdate-to implies --creationdate-from"') unless exists $opt->{creationdate_from};
        if ($opt->{creationdate_to} =~ m/^(\d{4})-(\d{2})-(\d{2})$/ ) {
            $to_epoch = Date_to_Time($1, $2, $3, 23, 59, 59);
        } else {
            $self->usage_error('--creationdate-to expects date in format "YYYY-MM-DD", got "'.$opt->{creationdate_to}.'"');
        }
    }
    if (defined $from_epoch and defined $to_epoch) {
        $opt->{creationdate_epochs}->{from} = $from_epoch;
        $opt->{creationdate_epochs}->{to} = $to_epoch;
        $self->usage_error('--date-to should have a date newer than --date-from') if ($from_epoch > $to_epoch);
    }
    if (exists $opt->{pronom_id}) {
        $self->usage_error("--pronom-id expects string which is conform to PUID structure as described in https://www.nationalarchives.gov.uk/aboutapps/pronom/puid.htm")
        unless ($opt->{pronom_id} =~ m/^(x-)?fmt\/[a-z0-9]+$/ );
    }
    if (exists $opt->{lzaid}) {
        my $rx_up = qr{[A-Za-z0-9_-]+}; # archive name & internal workflow
        my $rx_lw = qr{[a-z0-9_-]+};    # external workflow & external id
        $self->usage_error("--lzaid expects string which is conform to SLUBArchiv internal scheme")
            unless ($opt->{lzaid} =~ m/^$rx_up:$rx_up:$rx_up:$rx_lw:$rx_lw$/);
    }
    return 1;
}

sub execute {
    my ($self, $opt, $args) = @_;
    my $aips_query = SLUB::LZA::TA::Archivematica::Elasticsearch::PrepareQuery::prepare_aip_query($opt);
    my $response = SLUB::LZA::TA::Archivematica::Elasticsearch::query_elasticsearch_count(
        $SLUB::LZA::TA::config{elasticsearch_protocol},
        $SLUB::LZA::TA::config{elasticsearch_host},
        $SLUB::LZA::TA::config{elasticsearch_port},
        'aips', # indexname
        $aips_query, # query_hash ref
        {
            debug => $opt->{debug},
        }
    );
    if ($opt->{verbose}) {
        say np($response);
    } else {
        say $response;
    }
    return 1;
}

no Date::Calc;

1;
