package SLUB::LZA::TA::Command::init;
use SLUB::LZA::TA -command;
use v5.36;
use YAML qw(DumpFile);
use IO::Prompt::Tiny qw/prompt/;
use Crypt::Mode::CBC;
use Path::Tiny;
use namespace::autoclean -except => qr{^SLUB::LZA::TA::.*};

# VERSION

# ABSTRACT: init module for ta-tool

sub abstract { return "Initialize $0";}
sub description { return "Initialize $0, preparing config file using given options"}
sub opt_spec {
    my @global_opts= grep {
        defined $_->[0]
            and $_->[0] !~ m/^(elasticsearch-(host|port))|(http-only)|(permanent-volume)/;
    } SLUB::LZA::TA::common_global_opt_spec();
    my @local_opts = (
        [ "elasticsearch-host|e=s" => "host adress of Archivematica's ElasticSearch (mandatory)" ],
        [ "elasticsearch-port|E=s" => "port of Archivematica's ElasticSearch", { default => 9200 } ],
        [ "http-only" => "use HTTP instead HTTPS" ],
        [ "permanent-volume|P=s@" => "path[s] to volume where Archivematica stores AIPs permanently (mandatory, repeatable)"],
    );
    return (@global_opts, [], @local_opts);
}

sub validate_args { ## no critic (CognitiveComplexity::ProhibitExcessCognitiveComplexity)
    my ($self, $opt, $args) = @_;
    # no args allowed but options!
    $self->usage_error("No args allowed") if @$args;
    $self->usage_error("Missed mandatory parameter '--elasticsearch-host'") if (
        (!exists($opt->{elasticsearch_host}))
        and (!exists($opt->{help}))
    );
    $self->usage_error("Missed mandatory parameter '--permanent-volume'") if (
        (!exists($opt->{permanent_volume}))
            and (!exists($opt->{help}))
    );
    foreach my $vol (@{ $opt->{permanent_volume} }) {
        $self->usage_error("Parameter '--permanent-volume' expects an existing dir, got '$vol'!") unless -d $vol;
    }
    return 1;
}

sub execute {
    my ($self, $opt, $args) = @_;
    my %config;
    $config{elasticsearch_host} = $opt->{elasticsearch_host};
    $config{elasticsearch_port} = $opt->{elasticsearch_port};
    #$config{logdir} = $opt->{logdir};
    $config{http_only} = $opt->{http_only};
    $config{permanent_volume} = $opt->{permanent_volume};
    my $in_test = $ENV{TEST_ACTIVE};
    # next lines to fake interactive for testing
    local *IO::Prompt::Tiny::_is_interactive = *IO::Prompt::Tiny::_is_interactive;
    if ($in_test) {
        *IO::Prompt::Tiny::_is_interactive = sub {$in_test};
    }
    if (defined $opt->{debug}) {
        say STDERR "Is test active? ", $ENV{TEST_ACTIVE} ? "true" : "false";
        say STDERR "Is interactive? ", IO::Prompt::Tiny::_is_interactive() ? "true" : "false";
    }
    say STDERR "HINT: The password  stored in config file!";
    my $user = prompt('User:');
RETRY:
    my $passwd1 = prompt('Password:', -echo => "*");
    my $passwd2 = prompt('Password, again:', -echo => "*");
    if ($passwd1 ne $passwd2) {
        say "you typed different passwords, retry";
        goto RETRY;
    }
    $config{user} = "$user";
    $config{password} = SLUB::LZA::TA::Crypt::encrypt("$passwd1");
    if (defined $SLUB::LZA::TA::config_file) {
        if (defined $opt->{verbose}) {
            say "store config in $SLUB::LZA::TA::config_file";
        }
        my $file = path($SLUB::LZA::TA::config_file);
        $file->touch();
        $file->chmod("0600");
        my $fh = path($SLUB::LZA::TA::config_file)->filehandle({ exclusive => 0 }, ">");
        YAML::DumpFile($fh, \%config);
        $fh->close;
    }
    return 1;
}

1;
