package SLUB::LZA::TA::Command::report;
use SLUB::LZA::TA -command;
use v5.36;
use SLUB::LZA::TA::Archivematica::Elasticsearch;
use SLUB::LZA::TA::Archivematica::Elasticsearch::PrepareQuery;
use SLUB::LZA::TA::Output;

use Date::Calc qw(Date_to_Time Today Add_Delta_YM Add_Delta_YMD Day_of_Week);
use namespace::autoclean -except => qr{SLUB::LZA::TA::.*};

# VERSION

# ABSTRACT: report module for ta-tool

use constant {
    FULL       => 0,
    LDP => 1,
    NO_LDP     => 2,
};
use constant SETS => (
    "total",
    "LDP only",
    "no LDP",
);
use constant {
    NEW    => 0,
    UPDATE => 1,
    TOTAL  => 2
};
use constant AIPSTATE => (
    "first ingest",
    "AIP update",
    "all AIPs"
);

use constant {
    COUNT         => 0,
    FILES         => 1,
    SIZE          => 2,
    PAYLOAD_FILES => 3,
    PAYLOAD_SIZE  => 4,
};
use constant FLAVOURS => (
    "count of aips",
    "count of files",
    "size in B",
    "count of payload files",
    "payload size in B"
);

sub abstract { return "print AIP reports about Archival Information System (AIS)";}
my $base_cmd = "$0 report";
my $dummycmd = " "x length($base_cmd);
my $description=<<"DESCR";
Ask an AIS for some statistics about AIPs.

Overview:

$base_cmd [--daily| --weekly | --monthly | --yearly] [--workflow WORKFLOW]
          ([--with-ldp] [--with-filetypes])
$dummycmd --version
$dummycmd --help

Examples:

  * Report statistics of AIPs last month for workflow Kitodo
    '$base_cmd --monthly --workflow Kitodo'
  * Report statistics of AIPs last year
    '$base_cmd --yearly'

A printable PDF version could be generated using ff. commands:
    '$base_cmd | asciidoctor-pdf - > report.pdf'

HINT: If you want lists, use the 'search' command instead!
HINT: options '--with-ldp' and '--with-filetypes' should be used with care because the running time
      increases from O(n) to O(n²) if one option is used
      or increases to O(n³) if both options used
      (with n => count of AIPs)
HINT: ensure the ElasticSearch server allows to return >10.000 results if your archive is large

DESCR

sub description {
    return "$description"
}
sub opt_spec {
    my @global_opts= SLUB::LZA::TA::common_global_opt_spec();
    my @local_opts = (
        [ 'datemode' => hidden => {
            one_of => [
                [ 'daily|d'   => 'report based on last day'],
                [ 'weekly|W'  => 'report based on last week'],
                [ 'monthly|m' => 'report based on last month (default)' ],
                [ 'yearly|y'  => 'report based on last year' ],
                [ 'ldpyearly' => 'report based on last LDP year 01.11. - 31.10.'],
                [ 'complete|c'=> 'report based on all AIPs'],
                [ 'date-from=s' => 'report based on date range, beginning date in format "YYYY-MM-DD", implies "--date-to"'],
            ],
          },
        ],
        [ 'date-to=s' => 'report based on date range, end date in format "YYYY-MM-DD", implies "--date-from"'],
        [],
        [ 'output-format' => hidden => {
            one_of => [
                [ 'output-as-csv|C' => 'prints output as Comma Separated Values (CSV)' ],
                [ 'output-as-raw|R' => 'print raw hash output' ],
                [ 'output-as-rsv|r' => 'prints output as Raw Strings Values (RSV)' ],
                [ 'output-as-asciidoc|a' => 'prints output as AsciiDoc [default]' ],
            ],
        }
        ],
        [],
        [ 'with-ldp|l' => 'report with LDP, use it with caution, needs a lot of processing time'],
        [ 'with-filetypes|f' => 'report with filetypes, use it with caution, needs a lot of processing time'],
        [ 'workflow|w=s' => 'LZA internal workflow name (optional)'],
        [],
    );
    return (@global_opts, [], @local_opts);
}
sub validate_args { ## no critic (CognitiveComplexity::ProhibitExcessCognitiveComplexity)
    my ($self, $opt, $args) = @_;
    SLUB::LZA::TA::common_global_validate($self, $opt, $args);
    # no args allowed but options!
    $self->usage_error("No args allowed") if @$args;
    my ($cyear, $cmonth, $cday) = Today();
    my ($from_year, $from_month, $from_day);
    my ($to_year, $to_month, $to_day);
    unless (exists $opt->{datemode}) {
        $opt->{datemode} = "monthly";
        $opt->{monthly} = 1;
    }
    my %date_recipe;
    $date_recipe{daily} = sub {
        ($from_year, $from_month, $from_day) = Add_Delta_YMD($cyear, $cmonth, $cday, 0, 0, -1);
        ($to_year, $to_month, $to_day) = ($from_year, $from_month, $from_day);
    };
    $date_recipe{weekly} = sub {
        ($from_year, $from_month, $from_day) = Add_Delta_YMD($cyear, $cmonth, $cday, 0, 0, -Day_of_Week($cyear, $cmonth, $cday) - 6);
        ($to_year, $to_month, $to_day) = Add_Delta_YMD($from_year, $from_month, $from_day, 0, 0, 6);
    };
    $date_recipe{monthly} = sub {
        ($from_year, $from_month, $from_day) = Add_Delta_YM($cyear, $cmonth, 1, 0, -1);
        ($to_year, $to_month, $to_day) = Add_Delta_YMD($from_year, $from_month, $from_day, 0, 1, -1);
    };
    $date_recipe{yearly} = sub {
        ($from_year, $from_month, $from_day) = Add_Delta_YM($cyear, 1, 1, -1, 0);
        ($to_year, $to_month, $to_day) = Add_Delta_YMD($from_year, $from_month, $from_day, 1, 0, -1);
    };
    $date_recipe{complete} = sub {
        $from_year = 2015;
        $from_month = 1;
        $from_day = 1;
        $to_year = $cyear;
        $to_month = $cmonth;
        $to_day = $cday;
    };
    $date_recipe{ldpyearly} = sub {
        ($from_year, $from_month, $from_day) = Add_Delta_YM($cyear, 1, 1, -1, -2);
        ($to_year, $to_month, $to_day) = Add_Delta_YMD($from_year, $from_month, $from_day, 1, 0, -1);
    };
    $date_recipe{date_from} = sub {
        $self->usage_error('--date-from implies --date-to"') unless exists $opt->{date_to};
        if ($opt->{date_from} =~ m/^(\d{4})-(\d{2})-(\d{2})$/) {
            ($from_year, $from_month, $from_day) = ($1, $2, $3);
        } else {
            $self->usage_error('--date-from expects date in format "YYYY-MM-DD", got "' . $opt->{date_from} . '"');
        }
    };
    $date_recipe{date_to} = sub {
        $self->usage_error('--date-to implies --date-from"') unless exists $opt->{date_from};
        if ($opt->{date_to} =~ m/^(\d{4})-(\d{2})-(\d{2})$/) {
            ($to_year, $to_month, $to_day) = ($1, $2, $3);
        } else {
            $self->usage_error('--date-to expects date in format "YYYY-MM-DD", got "', $opt->{date_to} . '"');
        }
    };
    foreach my $key (keys %{ $opt } ) {
        $date_recipe{$key}->() if (defined $date_recipe{$key} and ref $date_recipe{$key} eq 'CODE');
    }
    $opt->{output_format} = 'output_as_asciidoc' unless (exists $opt->{output_format});
    my $from_epoch = Date_to_Time($from_year, $from_month, $from_day, 0, 0, 0);
    my $to_epoch = Date_to_Time($to_year, $to_month, $to_day, 0, 0, 0);
    $self->usage_error('--date-to should have a date newer than --date-from') if ($from_epoch > $to_epoch);
    printf STDERR "reporting for period %04u-%02u-%02u … %04u-%02u-%02u\n", $from_year, $from_month, $from_day, $to_year, $to_month, $to_day;
    say STDERR "HINT: the option '--with-ldp'       results in lot of processing time and network traffic, use it with care!" if (exists $opt->{with_ldp});
    say STDERR "HINT: the option '--with-filetypes' results in lot of processing time and network traffic, use it with care!" if (exists $opt->{with_filetypes});
    $opt->{creationdate_epochs}->{from} = $from_epoch;
    $opt->{creationdate_epochs}->{to} = $to_epoch;
    $opt->{creationdate_epochs}->{from_string} = sprintf("%04u-%02u-%02u", $from_year, $from_month, $from_day);
    $opt->{creationdate_epochs}->{to_string} = sprintf("%04u-%02u-%02u", $to_year, $to_month, $to_day);
    return 1;
}

sub _execute {
    my ($self, $opt, $args) = @_;
    my $aips_query;
    my $aips_response;
    #p($opt);
    # only index aips needed
    $aips_query = SLUB::LZA::TA::Archivematica::Elasticsearch::PrepareQuery::prepare_aip_query($opt);
    # next lines extend query with reporting
    $aips_query->{aggs} = {
        # add total aip size
        total_aip_size     =>
            { sum => {
                field => "size"
            }
            },
        # add total file count
        total_file_count   =>
            { sum => {
                field => "file_count"
            }
            },
        # add total payload size
        total_payload_size =>
            {
                sum => {
                    "script" => {
                        lang   => "painless",
                        source => <<"PAINLESS"
if (! doc['transferMetadata.bim:bag-info_dict.bim:Payload-Oxum.keyword'].empty) {
  return Math.floor(
    Double.parseDouble(
      doc['transferMetadata.bim:bag-info_dict.bim:Payload-Oxum.keyword'].value
    )
  )
} else {
  return 0
}
PAINLESS
                    }
                }
            },
        # add totol payload count
        total_payload_filecount =>
            {
                sum => {
                    "script" => {
                        lang   => "painless",
                        source => <<"PAINLESS"
if (! doc['transferMetadata.bim:bag-info_dict.bim:Payload-Oxum.keyword'].empty) {
  def s = doc['transferMetadata.bim:bag-info_dict.bim:Payload-Oxum.keyword'].value;
  def pos = s.lastIndexOf('.');
  return (
    Double.parseDouble(
      s.substring(pos+1)
    )
  );
} else {
  return 0
}
PAINLESS
                    }
                }
            }
    };
    $aips_query->{size}= 0; # only use aggregations
    $aips_response = SLUB::LZA::TA::Archivematica::Elasticsearch::query_elasticsearch(
        $SLUB::LZA::TA::config{elasticsearch_protocol},
        $SLUB::LZA::TA::config{elasticsearch_host},
        $SLUB::LZA::TA::config{elasticsearch_port},
        'aips',      # indexname
        $aips_query, # query_hash ref
        {
            debug => $opt->{debug},
        }
    );
    $aips_response->{from_date}=$opt->{creationdate_epochs}->{from_string};
    $aips_response->{to_date}=$opt->{creationdate_epochs}->{to_string};
    return $aips_response;
}

sub get_ldp_projects ($self, $opt, $args) { ## no critic (CognitiveComplexity::ProhibitExcessCognitiveComplexity)
    my $query = SLUB::LZA::TA::Archivematica::Elasticsearch::PrepareQuery::prepare_ldpprojects_query($self, $opt, $args);
    my $response = SLUB::LZA::TA::Archivematica::Elasticsearch::query_elasticsearch(
        $SLUB::LZA::TA::config{elasticsearch_protocol},
        $SLUB::LZA::TA::config{elasticsearch_host},
        $SLUB::LZA::TA::config{elasticsearch_port},
        'aips',      # indexname
        $query, # query_hash ref
        {
            debug => $opt->{debug},
        }
    );
    my %projects;
    foreach my $match (@{$response->{hits}->{hits}}) {
        foreach my $tme (@{ $match->{_source}->{transferMetadata} }) {
            my $dict = $tme->{'bim:bag-info_dict'};
            if (ref $dict eq 'ARRAY') {
                foreach my $e (@{ $dict }) {
                    my $project = $e->{'bim:LDP-project'};
                    $projects{$project} = 1;
                }
            } elsif (ref $dict eq 'HASH') {
                my $project = $dict->{'bim:LDP-project'};
                $projects{$project} = 1;
            }
        }
    }
    my @ret = sort keys %projects;
    return @ret;
}

sub get_filestypes_by_aips($self, $opt, $args) {
    my $aips_query = SLUB::LZA::TA::Archivematica::Elasticsearch::PrepareQuery::prepare_aip_query($opt);
    $aips_query->{"_source"} = {
        "includes" => 'uuid'
    };

    my $aips_response = SLUB::LZA::TA::Archivematica::Elasticsearch::query_elasticsearch(
        $SLUB::LZA::TA::config{elasticsearch_protocol},
        $SLUB::LZA::TA::config{elasticsearch_host},
        $SLUB::LZA::TA::config{elasticsearch_port},
        'aips',      # indexname
        $aips_query, # query_hash ref
        {
            debug => $opt->{debug},
        }
    );
    my @aips = map { $_->{_source}->{uuid} } @{ $aips_response->{hits}->{hits} };
    my %results;
    foreach my $aip (@aips) {
        my $files_query = {
            query     => {
                bool => {
                    must =>
                        [
                            {
                                match => { "AIPUUID" => $aip
                                    #"METS.amdSec.mets:amdSec_dict.mets:techMD_dict.mets:mdWrap_dict.mets:xmlData_dict.premis:object_dict.premis:formatRegistry_dict.premis:formatRegistryKey" => "$pronom_id"
                                    #"premis:formatRegistryKey" => "$pronom_id"
                                }
                            }
                        ],
                }
            },
            "size"    => 10000,
            # fields not supported in ES6, therefore we must use _source!
            "_source" => {
                "includes" => [
                    'fileExtension',
                    join(".", qw(
                    METS
                        amdSec
                        mets:amdSec_dict
                        mets:techMD_dict
                        mets:mdWrap_dict
                        mets:xmlData_dict
                        premis:object_dict
                        premis:objectCharacteristics_dict
                        premis:size
                    )
                    ),
                    join(".", qw(
                        METS
                        amdSec
                        mets:amdSec_dict
                        mets:techMD_dict
                        mets:mdWrap_dict
                        mets:xmlData_dict
                        premis:object_dict
                        premis:objectCharacteristics_dict
                        premis:format_dict
                        premis:formatRegistry_dict
                        premis:formatRegistryKey
                    )
                    ),
                ]
            }
        };
        my $files_response = SLUB::LZA::TA::Archivematica::Elasticsearch::query_elasticsearch(
            $SLUB::LZA::TA::config{elasticsearch_protocol},
            $SLUB::LZA::TA::config{elasticsearch_host},
            $SLUB::LZA::TA::config{elasticsearch_port},
            'aipfiles',   # indexname
            $files_query, # query_hash ref
            {
                debug => $opt->{debug},
            }
        );
        foreach my $file_response (@{ $files_response->{hits}->{hits} }) {
            my $pronom_id = $file_response->{_source}
                ->{'METS'}
                ->{'amdSec'}
                ->{'mets:amdSec_dict'}
                ->{'mets:techMD_dict'}
                ->{'mets:mdWrap_dict'}
                ->{'mets:xmlData_dict'}
                ->{'premis:object_dict'}
                ->{'premis:objectCharacteristics_dict'}
                ->{'premis:format_dict'}
                ->{'premis:formatRegistry_dict'}
                ->{'premis:formatRegistryKey'};
            my $size = $file_response->{_source}
                ->{'METS'}
                ->{'amdSec'}
                ->{'mets:amdSec_dict'}
                ->{'mets:techMD_dict'}
                ->{'mets:mdWrap_dict'}
                ->{'mets:xmlData_dict'}
                ->{'premis:object_dict'}
                ->{'premis:objectCharacteristics_dict'}
                ->{'premis:size'};
            my $file_extension = $file_response->{_source}->{fileExtension} // "(no extension)";
            #my %tmp;
            #$tmp{pronom_id} = $pronom_id;
            #$tmp{size} = $size;
            #$tmp{file_extension} = $file_extension;
            $results{pronom_id}->{$pronom_id}->{(FLAVOURS)[SIZE]} += $size;
            $results{pronom_id}->{$pronom_id}->{(FLAVOURS)[FILES]}++;
            $results{file_extension}->{$file_extension}->{(FLAVOURS)[SIZE]} += $size;
            $results{file_extension}->{$file_extension}->{(FLAVOURS)[FILES]}++;
        }
    }
    return \%results;
}

sub filtered_set_for_ldp($set, $newhashref) {
    delete $newhashref->{only_ldp};
    delete $newhashref->{no_ldp};
    if ($set eq (SETS)[LDP]) {$newhashref->{only_ldp} = 1}
    elsif ($set eq (SETS)[NO_LDP]) {$newhashref->{no_ldp} = 1;}
    else { # full, do not filter for ldp
    }
    return 1;
}

sub filtered_set_for_aipstate($aip_state, $newhashref) {
    if ($aip_state eq (AIPSTATE)[NEW]) {$newhashref->{only_new} = 1;}
    elsif ($aip_state eq (AIPSTATE)[UPDATE]) {$newhashref->{only_updated} = 1;}
    return 1;
}

sub set_flavour_results_for_filetypes($self, $opt, $args, $results_ref, $newhash_ref, $aip_state, $set) {
    if (exists($opt->{with_filetypes})) {
        my $filetypes_hashref = get_filestypes_by_aips($self, $newhash_ref, $args);
        foreach my $file_extension (sort keys %{$filetypes_hashref->{file_extension}}) {
            $results_ref->{flavour}->{(FLAVOURS)[FILES]}->{$aip_state}->{$set}->{sprintf "%20s %10s", "file extension", $file_extension} = $filetypes_hashref->{file_extension}->{$file_extension}->{(FLAVOURS)[FILES]};
            $results_ref->{flavour}->{(FLAVOURS)[SIZE]}->{$aip_state}->{$set}->{sprintf "%20s %10s", "file extension", $file_extension} = $filetypes_hashref->{file_extension}->{$file_extension}->{(FLAVOURS)[SIZE]};
        }
        foreach my $pronom_id (sort keys %{$filetypes_hashref->{pronom_id}}) {
            $results_ref->{flavour}->{(FLAVOURS)[FILES]}->{$aip_state}->{$set}->{sprintf "%20s %10s", "pronom id", $pronom_id} = $filetypes_hashref->{pronom_id}->{$pronom_id}->{(FLAVOURS)[FILES]};
            $results_ref->{flavour}->{(FLAVOURS)[SIZE]}->{$aip_state}->{$set}->{sprintf "%20s %10s", "pronom id", $pronom_id} = $filetypes_hashref->{pronom_id}->{$pronom_id}->{(FLAVOURS)[SIZE]};
        }
    }
    return 1;
}

sub set_ldpflavour_results_for_filetype($self, $opt, $args, $results_ref, $newhash_ref, $ldp_project, $aip_state) {
    if (exists($opt->{with_filetypes})) {
        my $filetypes_hashref = get_filestypes_by_aips($self, $newhash_ref, $args);
        foreach my $file_extension (sort keys %{$filetypes_hashref->{file_extension}}) {
            $results_ref->{ldp_project}->{$ldp_project}->{flavour}->{(FLAVOURS)[FILES]}->{$aip_state}->{sprintf "%20s %10s", "file extension", $file_extension} = $filetypes_hashref->{file_extension}->{$file_extension}->{(FLAVOURS)[FILES]};
            $results_ref->{ldp_project}->{$ldp_project}->{flavour}->{(FLAVOURS)[SIZE]}->{$aip_state}->{sprintf "%20s %10s", "file extension", $file_extension} = $filetypes_hashref->{file_extension}->{$file_extension}->{(FLAVOURS)[SIZE]};
        }
        foreach my $pronom_id (sort keys %{$filetypes_hashref->{pronom_id}}) {
            $results_ref->{ldp_project}->{$ldp_project}->{flavour}->{(FLAVOURS)[FILES]}->{$aip_state}->{sprintf "%20s %10s", "pronom id", $pronom_id} = $filetypes_hashref->{pronom_id}->{$pronom_id}->{(FLAVOURS)[FILES]};
            $results_ref->{ldp_project}->{$ldp_project}->{flavour}->{(FLAVOURS)[SIZE]}->{$aip_state}->{sprintf "%20s %10s", "pronom id", $pronom_id} = $filetypes_hashref->{pronom_id}->{$pronom_id}->{(FLAVOURS)[SIZE]};
        }
    }
    return 1;
}

sub execute($self, $opt, $args) {
    my %results;
    $results{date} = sprintf("%04u-%02u-%02u", Today());
    $results{package} = __PACKAGE__;
    $results{from} = $opt->{creationdate_epochs}->{from_string};
    $results{to} = $opt->{creationdate_epochs}->{to_string};
    my @ldp_projects = ($opt->{with_ldp}) ? get_ldp_projects($self, $opt, $args) : ();
    foreach my $aip_state (AIPSTATE) {
        my %newhash = %{$opt};
        filtered_set_for_aipstate($aip_state, \%newhash);
        foreach my $set (SETS) {
            filtered_set_for_ldp($set, \%newhash);
            my $res = _execute($self, \%newhash, $args);
            $results{flavour}->{(FLAVOURS)[COUNT]}->{$aip_state}->{$set}->{""} = $res->{hits}->{total};
            $results{flavour}->{(FLAVOURS)[SIZE]}->{$aip_state}->{$set}->{""} = $res->{aggregations}->{total_aip_size}->{value} * 1024 * 1024;
            $results{flavour}->{(FLAVOURS)[FILES]}->{$aip_state}->{$set}->{""} = $res->{aggregations}->{total_file_count}->{value};
            $results{flavour}->{(FLAVOURS)[PAYLOAD_SIZE]}->{$aip_state}->{$set}->{""} = $res->{aggregations}->{total_payload_size}->{value};
            $results{flavour}->{(FLAVOURS)[PAYLOAD_FILES]}->{$aip_state}->{$set}->{""} = $res->{aggregations}->{total_payload_filecount}->{value};
            set_flavour_results_for_filetypes($self, $opt, $args, \%results, \%newhash, $aip_state, $set);
        }
        undef %newhash;
        foreach my $ldp_project (@ldp_projects) {
            # only if @ldp_projects have size > 1
            %newhash = %{$opt};
            filtered_set_for_aipstate($aip_state, \%newhash);
            $newhash{only_ldp_project} = $ldp_project;
            my $res = _execute($self, \%newhash, $args);
            $results{ldp_project}->{$ldp_project}->{flavour}->{(FLAVOURS)[COUNT]}->{$aip_state}->{""} = $res->{hits}->{total};
            $results{ldp_project}->{$ldp_project}->{flavour}->{(FLAVOURS)[SIZE]}->{$aip_state}->{""} = $res->{aggregations}->{total_aip_size}->{value} * 1024 * 1024;
            $results{ldp_project}->{$ldp_project}->{flavour}->{(FLAVOURS)[FILES]}->{$aip_state}->{""} = $res->{aggregations}->{total_file_count}->{value};
            $results{ldp_project}->{$ldp_project}->{flavour}->{(FLAVOURS)[PAYLOAD_SIZE]}->{$aip_state}->{""} = $res->{aggregations}->{total_payload_size}->{value};
            $results{ldp_project}->{$ldp_project}->{flavour}->{(FLAVOURS)[PAYLOAD_FILES]}->{$aip_state}->{""} = $res->{aggregations}->{total_payload_filecount}->{value};
            set_ldpflavour_results_for_filetype($self, $opt, $args, \%results, \%newhash, $ldp_project, $aip_state)
        }
    }
    my ($headers, $table) = prepare_for_table(\%results, \@ldp_projects);
    print_humanreadable_report(\%results)                if ($opt->{output_format} eq 'output_as_asciidoc');
    SLUB::LZA::TA::Output::RSV::print_results($table)    if ($opt->{output_format} eq 'output_as_rsv');
    SLUB::LZA::TA::Output::CSV::print_results($table)    if ($opt->{output_format} eq 'output_as_csv');
    SLUB::LZA::TA::Output::Raw::print_results(\%results) if ($opt->{output_format} eq 'output_as_raw');
    say STDERR "report is already sent to STDOUT.";
    return 1;
}

sub prepare_for_table($results, $ldp_projects) { ## no critic (CognitiveComplexity::ProhibitExcessCognitiveComplexity)
    my @table;
    foreach my $set (SETS) {
        foreach my $aip_state (AIPSTATE) {
            foreach my $flavour (FLAVOURS) { # count size file
                foreach my $filter (sort keys %{$results->{flavour}->{$flavour}->{$aip_state}->{$set}}) {
                    my $line;
                    $line->{_set} = $set;
                    $line->{_subset} = "";
                    $line->{_timespan_from} = $results->{from};
                    $line->{_timespan_to} = $results->{to};
                    $line->{flavour} = $flavour;
                    $line->{aip_state} = $aip_state;
                    $line->{filter} = $filter;
                    $line->{value} = $results->{flavour}->{$flavour}->{$aip_state}->{$set}->{$filter};
                    push @table, $line;
                }
            }
        }
    }
    foreach my $ldp_project (@{ $ldp_projects }) {
        foreach my $aip_state (AIPSTATE) {
            foreach my $flavour (FLAVOURS) {
                #$results{ldp_project}->{$ldp_project}->{flavour}->{(FLAVOURS)[FILES]}->{$aip_state}->{"file_extension $file_extension"}
                foreach my $filter (sort keys %{$results->{ldp_project}->{$ldp_project}->{flavour}->{$flavour}->{$aip_state}}) {
                    # count size file
                    my $line;
                    $line->{_set} = "LDP";
                    $line->{_subset} = $ldp_project;
                    $line->{_timespan_from} = $results->{from};
                    $line->{_timespan_to} = $results->{to};
                    $line->{flavour} = $flavour;
                    $line->{aip_state} = $aip_state;
                    $line->{filter} = $filter;
                    $line->{value} = $results->{ldp_project}->{$ldp_project}->{flavour}->{$flavour}->{$aip_state}->{$filter};
                    push @table, $line;
                }
            }
        }
    }
    my @headers = sort keys %{ $table[0] };
    return \@headers, \@table;
}

sub print_humanreadable_report ($results) { ## no critic (CognitiveComplexity::ProhibitExcessCognitiveComplexity)
    say <<"RPTHEADER";
:lang: en
:doctype: article
:date: $results->{date}
:generator: $0 ($results->{package})
:toc:

= Report from $results->{from} to $results->{to}
RPTHEADER
    say "== Complete archive\n";
    foreach my $flavour (sort keys %{$results->{flavour}}) {
        say "=== $flavour\n";
        foreach my $aip_state (AIPSTATE) {
            printf "*  %10s:\n", $aip_state;
            #use Data::Printer; p($results->{flavour}->{$flavour}->{$aip_state}); #die;
            foreach my $set (SETS) {
                foreach my $filter (sort keys %{$results->{flavour}->{$flavour}->{$aip_state}->{$set}}) {
#say "FILTER='$filter'";
                    if ($filter eq "") {
                        printf "**    %15s: %10u\n",
                            "$set",
                            $results->{flavour}->{$flavour}->{$aip_state}->{$set}->{""}
                        ;
                    } else {
                        printf "***    %25s: %10u\n",
                            $filter,
                            $results->{flavour}->{$flavour}->{$aip_state}->{$set}->{$filter}
                        ;
                    }
                }
            }
            say;
        }
    }
    foreach my $ldp_project (sort keys %{$results->{ldp_project}}) {
        say "== LDP project '$ldp_project'\n";
        foreach my $flavour (sort keys %{$results->{ldp_project}->{$ldp_project}->{flavour}}) {
            say "=== $flavour\n";
            foreach my $aip_state (AIPSTATE) {
                printf "*  %10s:\n", $aip_state;
                foreach my $filter (sort keys %{$results->{ldp_project}->{$ldp_project}->{flavour}->{$flavour}->{$aip_state}}) {
                    if ($filter eq "") {
                        printf "**    %15s: %10u\n",
                            "total",
                            $results->{ldp_project}->{$ldp_project}->{flavour}->{$flavour}->{$aip_state}->{$filter},
                        ;
                    } else {
                        printf "***  %25s: %10u\n",
                            $filter,
                            $results->{ldp_project}->{$ldp_project}->{flavour}->{$flavour}->{$aip_state}->{$filter},
                        ;
                    }
                }
                say;
            }
        }
    }
    return 1;
}

1;
