#!perl
use v5.36;
use Test::Cmd;
use Test::More tests =>11;
use Test::TA;
my $testapp = Test::TA::testapp();
my $host = Test::TA::testhost();
local $ENV{'HOME'} = $testapp->workpath();
# tests
subtest 'help check' => sub {
    plan tests => 9;
    foreach my $param ("help search", "search -h", "search --help") {
        $testapp->run(args => $param);
        #say "STDOUT:",$testapp->stdout;
        #say "STDERR:",$testapp->stderr;
        like($testapp->stdout, qr/this usage screen/, "ta-tool $param, stdout");
        like($testapp->stderr, qr/^$/, "ta-tool $param, stderr");
        is($? >> 8, 0, "ta-tool $param, exit status");
    }
};

subtest 'check cli params' => sub {
    plan tests => 3;
    SKIP: {
        skip "testhost '$host' not available", 3 unless Test::TA::is_testhost_available();
        $testapp->run(args => 'search --lzaid SLUB:LZA:goobi:1 --debug');
        #say "STDOUT:",$testapp->stdout;
        #say "STDERR:",$testapp->stderr;
        like($testapp->stdout, qr//, 'ta-tool search by lzaid, wrong format, stdout');
        like($testapp->stderr, qr/^$.*Error: --[a-z].* expects.*Usage/s, 'ta-tool search by lzaid, wrong format, stderr');
        isnt($? >> 8, 0, 'ta-tool search by lzaid, wrong format, exit status');
    };
};

subtest 'elastic search non-existing lza id' => sub {
    plan tests => 6;
    SKIP: {
        skip "testhost '$host' not available", 6 unless Test::TA::is_testhost_available();
        $testapp->run(args => 'search --lzaid SLUB:LZA:Goobi:goobi:1 --debug');
        #say "STDOUT:",$testapp->stdout;
        #say "STDERR:",$testapp->stderr;
        like($testapp->stdout, qr/\A\s*\Z/s, 'ta-tool search by lzaid, no match, stdout');
        like($testapp->stderr, qr/\A\@must.*\@should.*query:.*found 0 results\Z/s, 'ta-tool search by lzaid, no match, stderr');
        is($? >> 8, 0, 'ta-tool search by lzaid, no match, exit status');

        $testapp->run(args => 'search --lzaid SLUB:LZA:Goobi:goobi:1');
        like($testapp->stdout, qr/\A\s*\Z/s, 'ta-tool search by lzaid, no match, stdout');
        like($testapp->stderr, qr/^found 0 results$/, 'ta-tool search by lzaid, no match, stderr');
        is($? >> 8, 0, 'ta-tool search by lzaid, no match, exit status');
    };
};

subtest 'elastic search existing lza id' => sub {
    plan tests => 6;
    SKIP: {
        skip "testhost '$host' not available", 6 unless Test::TA::is_testhost_available();
        $testapp->run(args => 'search --lzaid SLUB:LZA:testworkflow:testcases:test-sip_2023-01-01-11_11-11_11111 --debug');
        #say "STDOUT:",$testapp->stdout;
        #say "STDERR:",$testapp->stderr;
        like($testapp->stdout, qr/\A"aipid"\n"47daf26a-02a1-4776-a8db-de04c6708bd2"\Z/s, 'ta-tool search by lzaid, existing, stdout');
        like($testapp->stderr, qr/\A\@must.*\@should.*query:.*found 1 result\nreporting results output as csv\Z/s, 'ta-tool search by lzaid, existing, stderr');
        is($? >> 8, 0, 'ta-tool search by lzaid, no match, exit status');

        $testapp->run(args => 'search --lzaid SLUB:LZA:testworkflow:testcases:test-sip_2023-01-01-11_11-11_11111');
        #say "STDOUT:",$testapp->stdout;
        #say "STDERR:",$testapp->stderr;
        like($testapp->stdout, qr/\A"aipid"\n"47daf26a-02a1-4776-a8db-de04c6708bd2"\Z/s, 'ta-tool search by lzaid, existing, stdout');
        like($testapp->stderr, qr/\Afound 1 result\nreporting results output as csv\Z/s, 'ta-tool search by lzaid, existing, stderr');
        is($? >> 8, 0, 'ta-tool search by lzaid, existing, exit status');
    };
};

subtest 'elastic search existing dc entry' => sub {
    plan tests => 3;
    SKIP: {
        skip "testhost '$host' not available", 3 unless Test::TA::is_testhost_available();
        $testapp->run(args => 'search --descriptive "Retrodigitalisierte Fotosn"'); #SLUB:LZA:testworkflow:testcases:test-sip_2023-01-01-11_11-11_11111');
        #say "STDOUT:",$testapp->stdout;
        #say "STDERR:",$testapp->stderr;
        like($testapp->stdout, qr/\A"aipid"\n"6ae5f1f0-5306-404f-8fb7-445f82416932"\n"9a254f9d-1c19-478a-aacb-43be3a3c7d9d"\n.*"4f7516e4-23c8-4750-a411-91359fb7a869"\Z/s, 'ta-tool search by dc, existing, stdout');
        like($testapp->stderr, qr/\Afound 30 results\nreporting results 1 … 10 output as csv\Z/s, 'ta-tool search by dc, existing, stderr');
        is($? >> 8, 0, 'ta-tool search by lzaid, existing, exit status');
    };
};

subtest 'check output formats' => sub {
    plan tests => 9;
    SKIP: {
        skip "testhost '$host' not available", 9 unless Test::TA::is_testhost_available();
        $testapp->run(args => 'search --output-as-raw --lzaid SLUB:LZA:testworkflow:testcases:test-sip_2023-01-01-11_11-11_11111 ');
        #say "STDOUT:",$testapp->stdout;
        #say "STDERR:",$testapp->stderr;
        like($testapp->stdout, qr/\A\{\n\s*hits\s*\{\n\s*hits\s*\[\n/s, 'ta-tool search, raw output, stdout');
        like($testapp->stderr, qr/\Afound 1 result\nreporting results output as raw\Z/s, 'ta-tool search, raw output, stderr');
        is($? >> 8, 0, 'ta-tool search, raw output, exit status');

        $testapp->run(args => 'search --output-as-csv --lzaid SLUB:LZA:testworkflow:testcases:test-sip_2023-01-01-11_11-11_11111 --with-lzaid');
        #say "STDOUT:",$testapp->stdout;
        #say "STDERR:",$testapp->stderr;
        like($testapp->stdout, qr/\A"aipid","slubarchiv_lzaid"\n"47daf26a-02a1-4776-a8db-de04c6708bd2","SLUB:LZA:testworkflow:testcases:test-sip_2023-01-01-11_11-11_11111"\Z/s, 'ta-tool search, CSV output, stdout');
        like($testapp->stderr, qr/\Afound 1 result\nreporting results output as csv\Z/s, 'ta-tool search, CSV output, stderr');
        is($? >> 8, 0, 'ta-tool search, CSV output, exit status');

        $testapp->run(args => 'search --output-as-rsv --lzaid SLUB:LZA:testworkflow:testcases:test-sip_2023-01-01-11_11-11_11111 --with-lzaid');
        #say "STDOUT:",$testapp->stdout;
        #say "STDERR:",$testapp->stderr;
        like($testapp->stdout, qr/\Aaipid\x{ff}slubarchiv_lzaid\x{ff}\x{fd}47daf26a-02a1-4776-a8db-de04c6708bd2\x{ff}SLUB:LZA:testworkflow:testcases:test-sip_2023-01-01-11_11-11_11111\x{ff}\x{fd}\Z/s, 'ta-tool search, RSV output, stdout');
        like($testapp->stderr, qr/\Afound 1 result\nreporting results output as rsv\Z/s, 'ta-tool search, RSV output, stderr');
        is($? >> 8, 0, 'ta-tool search, RSV output, exit status');
    };
};

subtest 'check with-options' => sub {
    plan tests => 57;
    SKIP: {
        skip "testhost '$host' not available", 57 unless Test::TA::is_testhost_available();
        foreach my $with (qw(
            External-Description
            External-Identifier
            LDP-collection
            LDP-funder
            LDP-lender
            LDP-project
            SLUBArchiv-archivalValueDescription
            SLUBArchiv-exportToArchiveDate
            SLUBArchiv-externalId
            SLUBArchiv-externalIsilId
            SLUBArchiv-externalWorkflow
            SLUBArchiv-hasConservationReason
            SLUBArchiv-lzaId
            SLUBArchiv-migrated-AIP
            SLUBArchiv-origin-AIS
            SLUBArchiv-previous-AIP
            SLUBArchiv-rightsVersion
            SLUBArchiv-sipVersion
            Title
        )) {
            my $with_opt = "--with-" . lc($with);
            my $head = lc($with);
            $head =~ s/-/_/g;
            $testapp->run(args => 'search --output-as-csv --lzaid SLUB:LZA:testworkflow:testcases:test-sip_2023-01-01-11_11-11_11111 ' . $with_opt);
            #say "STDOUT:",$testapp->stdout;
            #say "STDERR:",$testapp->stderr;
            like($testapp->stdout, qr/\A"aipid","$head"\n"47daf26a-02a1-4776-a8db-de04c6708bd2",("[A-Za-zäöüßÄÖÜ0-9_:. -]*")|\d+(\.\d+)?\Z/s, 'ta-tool search, check "' . $with_opt . '", stdout');
            like($testapp->stderr, qr/\Afound 1 result\nreporting results output as csv\Z/s, 'ta-tool search, check "' . $with_opt . '", stderr');
            is($? >> 8, 0, 'ta-tool search, check "' . $with_opt . '", exit status');
        }
    };
};

subtest 'search by files' => sub {
    plan tests => 3;
    SKIP: {
        skip "testhost '$host' not available", 3 unless Test::TA::is_testhost_available();
        $testapp->run(args => 'search --pronom-id fmt/101');
        #say "STDOUT:",$testapp->stdout;
        #say "STDERR:",$testapp->stderr;
        like($testapp->stdout, qr/\A"aipid"\n.*"47daf26a-02a1-4776-a8db-de04c6708bd2"/s, 'ta-tool search by pronom-id, existing, stdout');
        like($testapp->stderr, qr/\Afound 166 results\nreporting results 1 … 10 output as csv\Z/s, 'ta-tool search by pronom-id, existing, stderr');
        is($? >> 8, 0, 'ta-tool search by pronom-id, existing, exit status');
    };
};

subtest 'check paginations' => sub {
    plan tests => 6;
    SKIP: {
        skip "testhost '$host' not available", 6 unless Test::TA::is_testhost_available();
        $testapp->run(args => 'search --pronom-id fmt/101 --with-result-index --output-as-csv --startrecord 2 --maxrecords 1');
        #say "STDOUT:",$testapp->stdout;
        #say "STDERR:",$testapp->stderr;
        like($testapp->stdout, qr/\A"idx","aipid"\n2,"2ed0c682-4565-4320-8247-6da0cc0642f1"\Z/s, 'ta-tool search by pronom-id, existing, stdout');
        like($testapp->stderr, qr/\Afound 166 results\nreporting results output as csv\Z/s, 'ta-tool search by pronom-id, existing, stderr');
        is($? >> 8, 0, 'ta-tool search by pronom-id, existing, exit status');

        $testapp->run(args => 'search --pronom-id fmt/101 --with-result-index --output-as-csv --startrecord 3 --maxrecords 2');
        #say "STDOUT:",$testapp->stdout;
        #say "STDERR:",$testapp->stderr;
        like($testapp->stdout, qr/\A"idx","aipid"\n3,"5687a403-673d-45c1-864f-f7f22a7aadb9"\n4,"db256475-10bd-40ee-83cc-dab8424346ba"\Z/s, 'ta-tool search by pronom-id, existing, stdout');
        like($testapp->stderr, qr/\Afound 166 results\nreporting results 3 … 4 output as csv\Z/s, 'ta-tool search by pronom-id, existing, stderr');
        is($? >> 8, 0, 'ta-tool search by pronom-id, existing, exit status');
    };
};

subtest 'search dates' => sub {
    plan tests => 6;
    SKIP: {
        skip "testhost '$host' not available", 6 unless Test::TA::is_testhost_available();
        $testapp->run(args => 'search --creationdate 2023-09-05');
        #say "STDOUT:",$testapp->stdout;
        #say "STDERR:",$testapp->stderr;
        like($testapp->stdout, qr/\A"aipid"\n"47daf26a-02a1-4776-a8db-de04c6708bd2"\n"e8bfdec5-d201-42d6-b293-94bb1668a7b6"\Z/s, 'ta-tool search date, existing, stdout');
        like($testapp->stderr, qr/\Afound 2 results\nreporting results 1 … 2 output as csv/s, 'ta-tool search date, existing, stderr');
        is($? >> 8, 0, 'ta-tool search date, existing, exit status');

        $testapp->run(args => 'search --creationdate-from 2023-09-05 --creationdate-to 2023-12-22');
        #say "STDOUT:",$testapp->stdout;
        #say "STDERR:",$testapp->stderr;
        like($testapp->stdout, qr/\A"aipid"\n"47daf26a-02a1-4776-a8db-de04c6708bd2"\n/s, 'ta-tool search dates, existing, stdout');
        like($testapp->stderr, qr/\Afound 25 results\nreporting results 1 … 10 output as csv/s, 'ta-tool search dates, existing, stderr');
        is($? >> 8, 0, 'ta-tool search dates, existing, exit status');
    };
};

subtest 'overrides' => sub {
    plan tests => 3;
    SKIP: {
        skip "testhost '$host' not available", 3 unless Test::TA::is_testhost_available();
        $testapp->run(args => 'search --elasticsearch-host localhost --creationdate-from 2023-09-05 --creationdate-to 2023-12-22');
        #say "STDOUT:",$testapp->stdout;
        #say "STDERR:",$testapp->stderr;
        like($testapp->stdout, qr/^$/s, 'ta-tool search dates, existing, override, stdout');
        like($testapp->stderr, qr/^Hint: override local config with/s, 'ta-tool search dates, existing, override stderr');
        is($? >> 8, 111, 'ta-tool search dates, existing, override, exit status');
    };
};

1;
