/* 
2014-2018 by Andreas Romeyke (SLUB Dresden)

The code is partially based on https://code.google.com/p/clamavj/source/browse/trunk/src/main/java/com/philvarner/clamavj/ClamScan.java?r=2
 https://github.com/vrtadmin/clamav-devel/blob/master/clamdscan/client.c and 
 https://github.com/vrtadmin/clamav-devel/blob/master/clamdscan/proto.c

 and therefore licensed under apache 2.0.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.slub.rosetta.dps.repository.plugin;


import com.exlibris.core.infra.common.exceptions.logging.ExLogger;
import com.exlibris.dps.repository.plugin.virusChcek.VirusCheckPluginV2;

import java.io.DataOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketException;
import java.util.HashMap;
import java.util.Map;

/**
 * SLUBVirusCheckClamAVPlugin
 * <p/>
 * ClamScan, should use clamdscan variant to avoid initialization overhead
 * <p/>
 * clamd-client opens a TCP-connection, see p18 in clamdoc.pdf
 * or source at https://github.com/vrtadmin/clamav-devel/blob/master/clamdscan/client.c
 * or source at https://github.com/vrtadmin/clamav-devel/blob/master/clamdscan/proto.c
 * code could also be copied from https://code.google.com/p/clamavj/source/browse/trunk/src/main/java/com/philvarner/clamavj/ClamScan.java?r=2
 *
 * @author andreas.romeyke@slub-dresden.de (Andreas Romeyke)
 * @see com.exlibris.dps.repository.plugin.virusChcek.VirusCheckPluginV2
 */
public class SLUBVirusCheckClamAVPlugin implements VirusCheckPluginV2 {
    //private static final ExLogger log = ExLogger.getExLogger(SLUBVirusCheckClamAVPlugin.class);
    private static final int DEFAULT_CHUNK_SIZE = 4096;
    private static final byte[] INSTREAM = "zINSTREAM\0".getBytes();
    private static final byte[] VERSION = "zVERSION\0".getBytes();
    private static final String RESPONSEOK = "stream: OK";
    private static final String FOUND_SUFFIX = "FOUND";
    private static final String STREAM_PREFIX = "stream: ";

    private static final ExLogger log = ExLogger.getExLogger(SLUBVirusCheckClamAVPlugin.class);
    private int timeout;
    private String host;
    private int port;
    /* Status:
     * @return 0 if last scan passed (means: virus free) -> PASSED
     * @return 1 if last scan found a virus              -> FAILED
     * @return >1 if last scan result is undetermined    -> UNDETERMINED
     */
    private enum Status {PASSED, FAILED, UNDETERMINED}; /* order is important, because we use .ordinal() in return code */
    private Status status = Status.UNDETERMINED;
    private String signature = "";
    /** constructor */
    public SLUBVirusCheckClamAVPlugin() {
        //log.info("SLUBVirusCheckPlugin instantiated with host=" + host + " port=" + port + " timeout=" + timeout);
        System.out.println("SLUBVirusCheckPlugin instantiated");
    }
    /** init params to configure the plugin via xml forms
     * @param initp parameter map
     */
    public void initParams(Map<String, String> initp) {
        this.host = initp.get("host");
        this.port = Integer.parseInt(initp.get("port"));
        this.timeout = Integer.parseInt(initp.get("timeout"));
        log.info("SLUBVirusCheckPlugin instantiated with host=" + host + " port=" + port + " timeout=" + timeout);
        //System.out.println("SLUBVirusCheckPlugin instantiated with host=" + host + " port=" + port + " timeout=" + timeout);
    }
    /** stand alone check, main file to call local installed clamd
     * @param args list of files which should be scanned
     */
    public static void main(String[] args) {

        SLUBVirusCheckClamAVPlugin plugin = new SLUBVirusCheckClamAVPlugin();
        Map<String, String> initp = new HashMap<String, String>();
        initp.put( "host", "127.0.0.1");
        initp.put( "port", "3310");
        initp.put( "timeout", "60000");
        plugin.initParams( initp );
        System.out.println("Agent: " + plugin.getAgent());
        for (String file : args) {
            plugin.scan(file);
            System.out.println("RESULT: " + plugin.isVirusFree() + " SIGNATURE: " + plugin.getOutput());
        }
    }

    /** get host
     *
     * @return host
     */
    protected String getHost() {
        return this.host;
    }

    /** get port
     *
     * @return port number
     */
    protected int getPort() {
        return this.port;
    }

    /** get timeout
     *
     * @return timeout in ms
     */
    protected int getTimeOut() {
        return this.timeout;
    }

    /** get signature of last scanned file
     *
     * @return signature name
     */
    protected String getSignature() {
        return this.signature;
    }

    /** set signature of last scanned file
     *
     * @param signature signature of last scanned file
     */
    protected void setSignature(String signature) {
        this.signature = signature;
    }

    /** get status of last scan
     *
     * @return status of last scan
     */
    protected Status getStatus() {
        return status;
    }

    /** set status of last scan
     *
     * @param status status of last scan
     */
    protected void setStatus(Status status) {
        this.status = status;
    }

    /** helper to cat 'in' stream to 'dos' stream for socket communication
     *
     * @param in raw input stream 'in'
     * @param dos special outputstream 'dos' for socket communication special for clamd
     * @param buffer buffer to buffer cat
     * @throws IOException if something goes wrong
     */
    private void writeStreamToStream(InputStream in, DataOutputStream dos, byte[] buffer) throws IOException {
        int read;
        while ((read = in.read(buffer)) > 0) {
            dos.writeInt(read);
            dos.write(buffer, 0, read);
        }
        dos.writeInt(0);
    }

    /** opens a socket
     *
     * @return socket
     * @throws IOException if soemthing goes wrong
     */
    private Socket openSocket() throws IOException {
        // create a socket
        Socket socket = new Socket();
        try {
            socket.setSoTimeout(getTimeOut());
            socket.setKeepAlive(true);
            socket.setReuseAddress( true );
            socket.connect(new InetSocketAddress(getHost(), getPort()));
            log.debug( "socket has timeout of: " + socket.getSoTimeout() + " ms");
            log.debug( "socket has keepalive set: " + socket.getKeepAlive());
            log.debug( "socket has reuse adress set: " + socket.getReuseAddress());
        } catch (SocketException e) {
            log.warn( "Problems to open socket", e);
        }
        return socket;
    }

    /** close socket
     *
     * @param socket socket which should be closed
     * @param dos associated outputstream to socket
     */
    private void closeSocket(Socket socket, DataOutputStream dos) {
        if (dos != null) try {
            dos.close();
        } catch (IOException e) {
            log.error("exception closing DataOutputStream", e);
            //System.out.println("exception closing DOS " + e);
        }
        try {
            socket.close();
        } catch (IOException e) {
            log.error("exception closing socket", e);
            //System.out.println("exception closing socket " + e);
        }
    }

    private String read_socket_to_buffer(Socket socket, byte[] buffer) throws IOException {
        int read = socket.getInputStream().read(buffer);
        String response = "";
        if (read > 0) response = new String(buffer, 0, read);
        return response;
    }

    /**
     * calls a simple clamd command via socket
     *
     * @param socket opened socket
     * @throws IOException if something goes wrong
     */
    private String callSocketCommand_Version(Socket socket) throws IOException {
        DataOutputStream dos = null;
        String response = "";
        try {
            dos = new DataOutputStream(socket.getOutputStream());
            dos.write(SLUBVirusCheckClamAVPlugin.VERSION);
            byte[] buffer = new byte[DEFAULT_CHUNK_SIZE];
            dos.flush();
           response = read_socket_to_buffer(socket, buffer).trim();
        } finally {
            closeSocket(socket, dos);
        }
        return response;
    }



    /**
     * calls an extended clamd command via socket, which expects an additional data inputstream which should be sent
     *
     * @param socket opened socket
     * @param in     input stream which should be sent to clamd
     * @throws IOException if something goes wrong
     */
    private String callSocketCommand_Stream(Socket socket, InputStream in) throws IOException {
        DataOutputStream dos = null;
        String response = "";
        try {
            dos = new DataOutputStream(socket.getOutputStream());
            dos.write(SLUBVirusCheckClamAVPlugin.INSTREAM);
            byte[] buffer = new byte[DEFAULT_CHUNK_SIZE];
            writeStreamToStream(in, dos, buffer);
            dos.flush();
            response = read_socket_to_buffer(socket, buffer).trim();
        } finally {
            closeSocket(socket, dos);
        }
        return response;
    }

    /** scans a given file for viruses
     *
     * @param fileFullPath scans given file via clamd
     */
    public void scan(String fileFullPath) {
        try {
            Socket socket = openSocket();

            InputStream in = new FileInputStream(fileFullPath);
            // send stream
            String result = callSocketCommand_Stream(socket, in);
            in.close();
            log.debug( "Response: " + result);
            //System.out.println("Response: " + result);
            // parse return code

            if (RESPONSEOK.equals(result)) {
                setStatus(Status.PASSED);
                log.info("scan of file '" + fileFullPath + "' passed");
            } else if (result.endsWith(FOUND_SUFFIX)) {
                if (result.contains(".Exploit.CVE")) { // we want to ignore CVE results
                    setStatus(Status.UNDETERMINED);
                    log.info("scan of file '" + fileFullPath + "' possibly failed, check manually if should be ignored!");
                } else if (result.contains("Eicar-Signature")) { // we want to ignore EICAR
                    setStatus(Status.UNDETERMINED);
                    log.info("scan of file '" + fileFullPath + "' possibly failed, because EICAR sequence detected, check manually if should be ignored!");
                } else {
                    setStatus(Status.FAILED);
                    log.info("scan of file '" + fileFullPath + "' failed");
                }
                setSignature(result.substring(STREAM_PREFIX.length(), result.lastIndexOf(FOUND_SUFFIX) - 1));
            } else {
                setStatus(Status.UNDETERMINED);
                log.warn("clamd protocol not fully implemented, result='" + result + "'");
                //System.out.println("clamd protocol not fully implemented");
            }
        } catch (IOException e) {
            log.error("exception creation socket in scan(), clamd not available at host=" + host + "port=" + port, e);
            //System.out.println("exception creation socket, clamd not available at host=" + host + "port=" + port + " " + e);
            setStatus(Status.UNDETERMINED);
            setSignature("ERROR: clamd not available");
        }
    }

    /** outcome of virus check
     *
     * @return signature of last scan
     */
    public String getOutput() {
        return getSignature();
    }

    /** get clamd agent version and signature version calling clamd-command VERSION
     *
     * @return string with clamd version and signature version
     */
    public String getAgent() {
        try {
            // create a socket
            Socket socket = openSocket();
            return callSocketCommand_Version(socket);
        } catch (IOException e) {
            log.error("exception creation socket in getAgent(), clamd not available at host=" + host + "port=" + port, e);
            //System.out.println("exception creation socket, clamd not available at host=" + host + "port=" + port + " " + e);
            setStatus(Status.FAILED);
            setSignature("ERROR: clamd not available");
            return "ERROR: clamd not available";
        }
    }

    /** result of last scan
     *
     * @return 0 if last scan passed (means: virus free)
     *         1 if last scan found a virus
     *         >1 if last scan result is undetermined
     */
    public int isVirusFree() {
        return (getStatus().ordinal());
    }
}
